<?php

namespace App\Console\Commands;

use App\Exports\LicensesExport;
use Illuminate\Console\Command;
use Maatwebsite\Excel\Facades\Excel;

class ExportLicensesToExcel extends Command
{
    protected $signature = 'license:export-excel 
                            {--path= : مسار حفظ الملف (اختياري)}';

    protected $description = 'تصدير جميع التراخيص إلى ملف Excel';

    public function handle()
    {
        $this->info('تصدير التراخيص إلى ملف Excel...');
        $this->newLine();

        $path = $this->option('path');
        
        if (!$path) {
            $path = storage_path('app/licenses_export_' . date('Y-m-d_His') . '.xlsx');
        }

        try {
            $filename = basename($path);
            $directory = dirname($path);
            
            if (!$directory || $directory === '.' || $directory === $filename) {
                $directory = storage_path('app');
                $filename = 'licenses_export_' . date('Y-m-d_His') . '.xlsx';
            }
            
            // إنشاء المجلد إذا لم يكن موجوداً
            if (!is_dir($directory)) {
                mkdir($directory, 0755, true);
            }
            
            $fullPath = $directory . DIRECTORY_SEPARATOR . $filename;
            
            // تصدير الملف مباشرة
            $export = new LicensesExport();
            
            // استخدام Excel::store مع المسار الكامل
            $relativePath = str_replace(storage_path('app') . DIRECTORY_SEPARATOR, '', $fullPath);
            Excel::store($export, $relativePath, 'local');
            
            // التأكد من وجود الملف
            $storedPath = storage_path('app/' . $relativePath);
            if (file_exists($storedPath)) {
                $fullPath = $storedPath;
            } else {
                // محاولة أخرى
                $fullPath = storage_path('app/' . $filename);
            }
            
            $this->info('✅ تم تصدير التراخيص بنجاح!');
            $this->newLine();
            $this->info('📁 الملف: ' . $fullPath);
            $this->newLine();
            
            // عرض إحصائيات
            $licenseDb = app(\App\Services\LicenseDatabaseService::class);
            $allLicenses = $licenseDb->all();
            
            $usedCount = 0;
            $unusedCount = 0;
            
            foreach ($allLicenses as $license) {
                $status = $license['status'] ?? 'pending';
                if (in_array($status, ['active', 'expired', 'suspended'])) {
                    $usedCount++;
                } else {
                    $unusedCount++;
                }
            }
            
            $this->table(
                ['المعلومة', 'القيمة'],
                [
                    ['إجمالي التراخيص', count($allLicenses)],
                    ['مستعملة', $usedCount],
                    ['غير مستعملة', $unusedCount],
                    ['مسار الملف', $path],
                ]
            );
            
            return 0;
        } catch (\Exception $e) {
            $this->error('❌ فشل تصدير التراخيص: ' . $e->getMessage());
            return 1;
        }
    }
}

