<?php

namespace App\Console\Commands;

use App\Services\LicenseDatabaseService;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Carbon\Carbon;

class GenerateBulkLicenses extends Command
{
    protected $signature = 'license:generate-bulk 
                            {count=100 : عدد التراخيص المراد إنشاؤها}
                            {--expires= : تاريخ الانتهاء (Y-m-d) أو null للدائم}
                            {--users=10 : عدد المستخدمين المسموح لكل ترخيص}';

    protected $description = 'إنشاء عدة تراخيص دفعة واحدة';

    public function handle()
    {
        $count = (int)$this->argument('count');
        $expires = $this->option('expires');
        $maxUsers = (int)($this->option('users') ?? 10);

        if ($count < 1 || $count > 1000) {
            $this->error('عدد التراخيص يجب أن يكون بين 1 و 1000');
            return 1;
        }

        $this->info("إنشاء {$count} ترخيص...");
        $this->newLine();

        $licenseDb = app(LicenseDatabaseService::class);
        $bar = $this->output->createProgressBar($count);
        $bar->start();

        $successCount = 0;
        $failedCount = 0;
        $licenses = [];

        for ($i = 1; $i <= $count; $i++) {
            // إنشاء License Key
            $licenseKey = strtoupper(
                Str::random(4) . '-' . 
                Str::random(4) . '-' . 
                Str::random(4) . '-' . 
                Str::random(4)
            );

            // إنشاء Serial Number
            $serialNumber = 'SN-' . strtoupper(Str::random(12));

            // تاريخ الانتهاء
            $expiresAt = null;
            if ($expires && $expires !== 'null') {
                try {
                    $expiresAt = Carbon::parse($expires);
                } catch (\Exception $e) {
                    // تجاهل الخطأ واستخدام null
                }
            }

            $licenseData = [
                'license_key' => $licenseKey,
                'serial_number' => $serialNumber,
                'email' => null,
                'customer_name' => "ترخيص #{$i}",
                'status' => 'pending',
                'expires_at' => $expiresAt ? $expiresAt->format('Y-m-d H:i:s') : null,
                'max_users' => $maxUsers,
                'notes' => "تم إنشاؤه تلقائياً - {$count} تراخيص",
            ];

            if ($licenseDb->insert($licenseData)) {
                $successCount++;
                $licenses[] = [
                    'number' => $i,
                    'license_key' => $licenseKey,
                    'serial_number' => $serialNumber,
                ];
            } else {
                $failedCount++;
            }

            $bar->advance();
        }

        $bar->finish();
        $this->newLine(2);

        // عرض النتائج
        $this->info("✅ تم إنشاء {$successCount} ترخيص بنجاح!");
        if ($failedCount > 0) {
            $this->warn("⚠️  فشل إنشاء {$failedCount} ترخيص");
        }

        // حفظ التراخيص في ملف
        $this->saveToFile($licenses);

        $this->newLine();
        $this->info("📁 تم حفظ قائمة التراخيص في: storage/app/generated_licenses.txt");

        return 0;
    }

    protected function saveToFile(array $licenses): void
    {
        $filePath = storage_path('app/generated_licenses.txt');
        $content = "═══════════════════════════════════════════════════════════════\n";
        $content .= "  قائمة التراخيص المُنشأة\n";
        $content .= "  Generated Licenses List\n";
        $content .= "═══════════════════════════════════════════════════════════════\n\n";
        $content .= "تاريخ الإنشاء: " . now()->format('Y-m-d H:i:s') . "\n";
        $content .= "عدد التراخيص: " . count($licenses) . "\n\n";
        $content .= "═══════════════════════════════════════════════════════════════\n\n";

        foreach ($licenses as $license) {
            $content .= sprintf(
                "ترخيص #%d:\n  License Key: %s\n  Serial Number: %s\n\n",
                $license['number'],
                $license['license_key'],
                $license['serial_number']
            );
        }

        file_put_contents($filePath, $content);
    }
}






