<?php

namespace App\Console\Commands;

use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\StockAlert;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class GeneratePurchaseOrders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'stock:generate-purchase-orders {--supplier-id= : Specific supplier ID}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically generate purchase orders for products with pending stock alerts';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Generating purchase orders...');

        // Get pending stock alerts
        $alerts = StockAlert::with('product')
            ->where('status', 'pending')
            ->get();

        if ($alerts->isEmpty()) {
            $this->info('No pending stock alerts found.');
            return Command::SUCCESS;
        }

        // Group alerts by supplier (if product has preferred supplier)
        $alertsBySupplier = $alerts->groupBy(function ($alert) {
            // Try to get supplier from last purchase invoice
            $lastPurchase = $alert->product->purchaseInvoiceItems()
                ->with('purchaseInvoice.supplier')
                ->latest()
                ->first();
            
            return $lastPurchase?->purchaseInvoice?->supplier_id ?? 0; // 0 = no supplier
        });

        $ordersCreated = 0;

        DB::beginTransaction();
        try {
            foreach ($alertsBySupplier as $supplierId => $supplierAlerts) {
                if ($this->option('supplier-id') && $supplierId != $this->option('supplier-id')) {
                    continue;
                }

                // Calculate quantity to order (reorder level * 2 or minimum order quantity)
                $orderItems = [];
                foreach ($supplierAlerts as $alert) {
                    $product = $alert->product;
                    $quantityToOrder = max(
                        $product->reorder_level * 2, // Order 2x reorder level
                        $product->reorder_level - $product->stock_quantity + ($product->reorder_level * 2) // Ensure we have enough
                    );

                    $orderItems[] = [
                        'product_id' => $product->id,
                        'quantity' => $quantityToOrder,
                        'unit_price' => $product->purchase_price ?? 0,
                    ];
                }

                if (empty($orderItems)) {
                    continue;
                }

                // Create purchase order
                $purchaseOrder = PurchaseOrder::create([
                    'supplier_id' => $supplierId > 0 ? $supplierId : null,
                    'status' => 'draft',
                    'generation_type' => 'auto',
                    'expected_delivery_date' => now()->addDays(7), // Default 7 days
                    'notes' => 'Auto-generated purchase order from stock alerts',
                    'created_by' => 1, // System user
                ]);

                // Create order items
                foreach ($orderItems as $item) {
                    PurchaseOrderItem::create([
                        'purchase_order_id' => $purchaseOrder->id,
                        'product_id' => $item['product_id'],
                        'quantity' => $item['quantity'],
                        'unit_price' => $item['unit_price'],
                    ]);
                }

                $purchaseOrder->updateTotals();

                // Mark alerts as acknowledged
                foreach ($supplierAlerts as $alert) {
                    $alert->acknowledge();
                }

                $ordersCreated++;
                $this->info("Created purchase order: {$purchaseOrder->order_number} with " . count($orderItems) . " items");
            }

            DB::commit();
            $this->info("Successfully created {$ordersCreated} purchase order(s).");
            
            return Command::SUCCESS;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error("Error generating purchase orders: " . $e->getMessage());
            return Command::FAILURE;
        }
    }
}
