<?php

namespace App\Console\Commands;

use App\Models\Invoice;
use App\Mail\InvoiceReminder;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Carbon\Carbon;

class SendInvoiceReminders extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoices:send-reminders {--days=7 : Number of days before due date to send reminder}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send automatic reminders for due invoices';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $daysBeforeDue = (int) $this->option('days');
        $reminderDate = Carbon::now()->addDays($daysBeforeDue);

        // Get invoices that are due or will be due soon
        $invoices = Invoice::whereIn('status', ['final', 'partially_paid'])
            ->where('due_amount', '>', 0)
            ->whereHas('customer', function ($q) {
                $q->whereNotNull('email');
            })
            ->with(['customer', 'items.product'])
            ->get();

        $sentCount = 0;
        $failedCount = 0;

        foreach ($invoices as $invoice) {
            try {
                // Calculate days overdue
                $daysOverdue = Carbon::now()->diffInDays($invoice->created_at);
                
                // Check if reminder should be sent
                $shouldSend = true;
                
                // Check if invoice was already paid today (avoid sending if just paid)
                if ($invoice->updated_at->isToday() && $invoice->paid_amount > 0) {
                    $shouldSend = false;
                }

                if ($shouldSend && $invoice->customer && $invoice->customer->email) {
                    Mail::to($invoice->customer->email)
                        ->send(new InvoiceReminder($invoice, $daysOverdue));
                    
                    $sentCount++;
                    $this->info("Reminder sent for invoice #{$invoice->invoice_number} to {$invoice->customer->email}");
                }
            } catch (\Exception $e) {
                $failedCount++;
                $this->error("Failed to send reminder for invoice #{$invoice->invoice_number}: " . $e->getMessage());
            }
        }

        $this->info("Reminders sent: {$sentCount}, Failed: {$failedCount}");
        
        return Command::SUCCESS;
    }
}
