<?php

namespace App\Exports;

use App\Services\LicenseDatabaseService;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithStyles;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Color;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class LicensesExport implements FromCollection, WithHeadings, WithMapping, ShouldAutoSize, WithStyles, WithColumnWidths
{
    protected $licenseDb;

    public function __construct()
    {
        $this->licenseDb = app(LicenseDatabaseService::class);
    }

    public function collection()
    {
        $allLicenses = $this->licenseDb->all();
        
        // ترتيب التراخيص: المستعملة أولاً، ثم غير المستعملة
        usort($allLicenses, function($a, $b) {
            $statusOrder = ['active' => 1, 'expired' => 2, 'suspended' => 3, 'pending' => 4];
            $aOrder = $statusOrder[$a['status'] ?? 'pending'] ?? 5;
            $bOrder = $statusOrder[$b['status'] ?? 'pending'] ?? 5;
            
            if ($aOrder !== $bOrder) {
                return $aOrder <=> $bOrder;
            }
            
            return ($a['id'] ?? 0) <=> ($b['id'] ?? 0);
        });
        
        return collect($allLicenses);
    }

    public function headings(): array
    {
        return [
            'ID',
            'License Key',
            'Serial Number',
            'الحالة / Status',
            'مستعمل / Used',
            'Email',
            'اسم العميل / Customer Name',
            'Machine ID',
            'اسم الجهاز / Machine Name',
            'تاريخ التفعيل / Activated At',
            'تاريخ الانتهاء / Expires At',
            'عدد المستخدمين / Max Users',
            'ملاحظات / Notes',
            'تاريخ الإنشاء / Created At',
        ];
    }

    public function map($license): array
    {
        $status = $license['status'] ?? 'pending';
        $isUsed = in_array($status, ['active', 'expired', 'suspended']);
        
        $statusLabels = [
            'pending' => 'غير مستعمل / Not Used',
            'active' => 'مستعمل / Used (Active)',
            'expired' => 'مستعمل / Used (Expired)',
            'suspended' => 'مستعمل / Used (Suspended)',
        ];
        
        return [
            $license['id'] ?? '',
            $license['license_key'] ?? '',
            $license['serial_number'] ?? '',
            $statusLabels[$status] ?? $status,
            $isUsed ? 'نعم / Yes' : 'لا / No',
            $license['email'] ?? '',
            $license['customer_name'] ?? '',
            $license['machine_id'] ?? '',
            $license['machine_name'] ?? '',
            $license['activated_at'] ?? '',
            $license['expires_at'] ?? 'دائم / Permanent',
            $license['max_users'] ?? 10,
            $license['notes'] ?? '',
            $license['created_at'] ?? '',
        ];
    }

    public function columnWidths(): array
    {
        return [
            'A' => 8,   // ID
            'B' => 25,  // License Key
            'C' => 20,  // Serial Number
            'D' => 25,  // Status
            'E' => 15,  // Used
            'F' => 30,  // Email
            'G' => 25,  // Customer Name
            'H' => 30,  // Machine ID
            'I' => 25,  // Machine Name
            'J' => 18,  // Activated At
            'K' => 18,  // Expires At
            'L' => 15,  // Max Users
            'M' => 40,  // Notes
            'N' => 18,  // Created At
        ];
    }

    public function styles(Worksheet $sheet)
    {
        // تنسيق رأس الجدول
        $sheet->getStyle('A1:N1')->applyFromArray([
            'font' => [
                'bold' => true,
                'color' => ['rgb' => 'FFFFFF'],
                'size' => 12,
            ],
            'fill' => [
                'fillType' => Fill::FILL_SOLID,
                'startColor' => ['rgb' => '4472C4'],
            ],
            'alignment' => [
                'horizontal' => Alignment::HORIZONTAL_CENTER,
                'vertical' => Alignment::VERTICAL_CENTER,
            ],
        ]);

        // تحديد ارتفاع الصف الأول
        $sheet->getRowDimension(1)->setRowHeight(25);

        // تطبيق التنسيق على جميع الصفوف
        $highestRow = $sheet->getHighestRow();
        for ($row = 2; $row <= $highestRow; $row++) {
            $statusCell = $sheet->getCell('D' . $row)->getValue();
            
            // تلوين الصفوف حسب الحالة
            if (strpos($statusCell, 'Used') !== false) {
                // صفوف التراخيص المستعملة - لون فاتح
                $sheet->getStyle('A' . $row . ':N' . $row)->applyFromArray([
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'E2EFDA'], // أخضر فاتح
                    ],
                ]);
            } else {
                // صفوف التراخيص غير المستعملة - لون أبيض
                $sheet->getStyle('A' . $row . ':N' . $row)->applyFromArray([
                    'fill' => [
                        'fillType' => Fill::FILL_SOLID,
                        'startColor' => ['rgb' => 'FFFFFF'],
                    ],
                ]);
            }
            
            // محاذاة النصوص
            $sheet->getStyle('A' . $row . ':N' . $row)->applyFromArray([
                'alignment' => [
                    'vertical' => Alignment::VERTICAL_CENTER,
                ],
            ]);
        }

        // إضافة حدود للخلايا
        $sheet->getStyle('A1:N' . $highestRow)->applyFromArray([
            'borders' => [
                'allBorders' => [
                    'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                    'color' => ['rgb' => 'CCCCCC'],
                ],
            ],
        ]);

        // تجميد الصف الأول
        $sheet->freezePane('A2');

        return $sheet;
    }
}






