<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ActivityLogController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasAnyRole(['Admin', 'Super Admin'])) {
                abort(403, trans('messages.unauthorized_action'));
            }
            return $next($request);
        });
    }

    public function index(Request $request)
    {
        $query = ActivityLog::with('user');

        // Filter by user
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by action
        if ($request->filled('action')) {
            $query->where('action', $request->action);
        }

        // Filter by model type
        if ($request->filled('model_type')) {
            $query->where('model_type', $request->model_type);
        }

        // Filter by date range
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                  ->orWhere('route', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($userQuery) use ($search) {
                      $userQuery->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%");
                  });
            });
        }

        $logs = $query->latest()->paginate(50);
        $users = User::orderBy('name')->get();
        $actions = ActivityLog::distinct()->pluck('action')->sort();
        $modelTypes = ActivityLog::distinct()->whereNotNull('model_type')->pluck('model_type')->sort();

        return view('activity-logs.index', compact('logs', 'users', 'actions', 'modelTypes'));
    }

    public function show(ActivityLog $activityLog)
    {
        $activityLog->load('user');
        
        return view('activity-logs.show', compact('activityLog'));
    }

    public function destroy(ActivityLog $activityLog)
    {
        if (!auth()->user()->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $activityLog->delete();

        return redirect()->route('activity-logs.index')
            ->with('success', trans('messages.activity_log_deleted'));
    }

    public function clear(Request $request)
    {
        if (!auth()->user()->hasRole('Super Admin')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $query = ActivityLog::query();

        // Filter by date
        if ($request->filled('date_before')) {
            $query->whereDate('created_at', '<', $request->date_before);
        }

        $deleted = $query->delete();

        return redirect()->route('activity-logs.index')
            ->with('success', trans('messages.activity_logs_cleared', ['count' => $deleted]));
    }
}
