<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Product;
use App\Models\Payment;
use App\Models\PurchaseInvoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Admin|Super Admin');
    }

    public function index()
    {
        return view('analytics.index');
    }

    // Sales Trends Analysis
    public function salesTrends(Request $request)
    {
        $period = $request->get('period', 'monthly'); // daily, weekly, monthly, yearly
        $months = $request->get('months', 6); // Number of months to analyze
        
        $dateFrom = now()->subMonths($months)->startOfDay();
        $dateTo = now()->endOfDay();

        $invoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->get();

        $trends = [];
        $labels = [];
        $salesData = [];
        $profitData = [];
        $countData = [];

        switch ($period) {
            case 'daily':
                $grouped = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-m-d');
                });
                $current = $dateFrom->copy();
                while ($current <= $dateTo) {
                    $key = $current->format('Y-m-d');
                    $group = $grouped->get($key, collect());
                    $labels[] = $current->format('Y-m-d');
                    $salesData[] = $group->sum('total');
                    $profitData[] = $this->calculateProfit($group);
                    $countData[] = $group->count();
                    $current->addDay();
                }
                break;

            case 'weekly':
                $grouped = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-W');
                });
                $current = $dateFrom->copy()->startOfWeek();
                while ($current <= $dateTo) {
                    $key = $current->format('Y-W');
                    $group = $grouped->get($key, collect());
                    $labels[] = 'Week ' . $current->format('W');
                    $salesData[] = $group->sum('total');
                    $profitData[] = $this->calculateProfit($group);
                    $countData[] = $group->count();
                    $current->addWeek();
                }
                break;

            case 'monthly':
                $grouped = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y-m');
                });
                $current = $dateFrom->copy()->startOfMonth();
                while ($current <= $dateTo) {
                    $key = $current->format('Y-m');
                    $group = $grouped->get($key, collect());
                    $labels[] = $current->format('M Y');
                    $salesData[] = $group->sum('total');
                    $profitData[] = $this->calculateProfit($group);
                    $countData[] = $group->count();
                    $current->addMonth();
                }
                break;

            case 'yearly':
                $grouped = $invoices->groupBy(function ($invoice) {
                    return $invoice->created_at->format('Y');
                });
                $current = $dateFrom->copy()->startOfYear();
                while ($current <= $dateTo) {
                    $key = $current->format('Y');
                    $group = $grouped->get($key, collect());
                    $labels[] = $current->format('Y');
                    $salesData[] = $group->sum('total');
                    $profitData[] = $this->calculateProfit($group);
                    $countData[] = $group->count();
                    $current->addYear();
                }
                break;
        }

        // Calculate trend direction
        $trendDirection = 'stable';
        if (count($salesData) >= 2) {
            $recent = array_slice($salesData, -3);
            $previous = array_slice($salesData, -6, 3);
            $recentAvg = array_sum($recent) / count($recent);
            $previousAvg = count($previous) > 0 ? array_sum($previous) / count($previous) : $recentAvg;
            
            if ($recentAvg > $previousAvg * 1.1) {
                $trendDirection = 'up';
            } elseif ($recentAvg < $previousAvg * 0.9) {
                $trendDirection = 'down';
            }
        }

        return view('analytics.sales-trends', compact(
            'labels', 'salesData', 'profitData', 'countData', 
            'period', 'months', 'trendDirection'
        ));
    }

    // Sales Forecast
    public function salesForecast(Request $request)
    {
        $months = $request->get('months', 6);
        $forecastPeriods = $request->get('forecast_periods', 3);

        // Get historical data
        $dateFrom = now()->subMonths($months)->startOfMonth();
        $dateTo = now()->endOfMonth();

        $invoices = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->get();

        $historicalData = $invoices->groupBy(function ($invoice) {
            return $invoice->created_at->format('Y-m');
        })->map(function ($group) {
            return [
                'sales' => $group->sum('total'),
                'count' => $group->count(),
                'profit' => $this->calculateProfit($group),
            ];
        })->sortKeys();

        $labels = $historicalData->keys()->toArray();
        $salesValues = $historicalData->pluck('sales')->toArray();
        $countValues = $historicalData->pluck('count')->toArray();

        // Simple linear regression for forecasting
        $forecastLabels = [];
        $forecastSales = [];
        $forecastCount = [];

        if (count($salesValues) >= 2) {
            // Calculate trend
            $n = count($salesValues);
            $sumX = 0;
            $sumY = 0;
            $sumXY = 0;
            $sumX2 = 0;

            foreach ($salesValues as $index => $value) {
                $x = $index + 1;
                $sumX += $x;
                $sumY += $value;
                $sumXY += $x * $value;
                $sumX2 += $x * $x;
            }

            $slope = ($n * $sumXY - $sumX * $sumY) / ($n * $sumX2 - $sumX * $sumX);
            $intercept = ($sumY - $slope * $sumX) / $n;

            // Generate forecast
            $lastDate = Carbon::parse(end($labels))->startOfMonth();
            for ($i = 1; $i <= $forecastPeriods; $i++) {
                $forecastDate = $lastDate->copy()->addMonths($i);
                $forecastLabels[] = $forecastDate->format('M Y');
                $x = $n + $i;
                $forecastSales[] = max(0, $slope * $x + $intercept);
                
                // Forecast count using same method
                $countSlope = ($n * array_sum(array_map(function($i, $v) { return ($i+1) * $v; }, array_keys($countValues), $countValues)) - $sumX * array_sum($countValues)) / ($n * $sumX2 - $sumX * $sumX);
                $countIntercept = (array_sum($countValues) - $countSlope * $sumX) / $n;
                $forecastCount[] = max(0, round($countSlope * $x + $countIntercept));
            }
        }

        return view('analytics.sales-forecast', compact(
            'labels', 'salesValues', 'countValues',
            'forecastLabels', 'forecastSales', 'forecastCount',
            'months', 'forecastPeriods'
        ));
    }

    // Profitability Analysis
    public function profitability(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->subMonths(6)->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        $invoiceItems = InvoiceItem::whereHas('invoice', function ($q) use ($dateFrom, $dateTo) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final'])
                  ->whereBetween('created_at', [$dateFrom, $dateTo]);
            })
            ->with(['product', 'invoice'])
            ->get();

        // Product profitability
        $productProfitability = $invoiceItems->groupBy('product_id')->map(function ($group) {
            $product = $group->first()->product;
            $revenue = $group->sum('total');
            $cost = $group->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });
            $profit = $revenue - $cost;
            $margin = $revenue > 0 ? ($profit / $revenue) * 100 : 0;

            return [
                'product' => $product,
                'revenue' => $revenue,
                'cost' => $cost,
                'profit' => $profit,
                'margin' => $margin,
                'quantity' => $group->sum('quantity'),
            ];
        })->sortByDesc('profit')->take(20);

        // Category profitability
        $categoryProfitability = $invoiceItems->groupBy(function ($item) {
            return $item->product->category_id ?? 0;
        })->map(function ($group) {
            $category = $group->first()->product->category;
            $revenue = $group->sum('total');
            $cost = $group->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });
            $profit = $revenue - $cost;
            $margin = $revenue > 0 ? ($profit / $revenue) * 100 : 0;

            return [
                'category' => $category,
                'revenue' => $revenue,
                'cost' => $cost,
                'profit' => $profit,
                'margin' => $margin,
            ];
        })->sortByDesc('profit');

        // Monthly profitability trend
        $monthlyProfit = $invoiceItems->groupBy(function ($item) {
            return $item->invoice->created_at->format('Y-m');
        })->map(function ($group) {
            $revenue = $group->sum('total');
            $cost = $group->sum(function ($item) {
                return $item->quantity * ($item->product->purchase_price ?? 0);
            });
            return [
                'revenue' => $revenue,
                'cost' => $cost,
                'profit' => $revenue - $cost,
            ];
        })->sortKeys();

        $monthlyLabels = $monthlyProfit->keys()->toArray();
        $monthlyRevenue = $monthlyProfit->pluck('revenue')->toArray();
        $monthlyCost = $monthlyProfit->pluck('cost')->toArray();
        $monthlyProfitData = $monthlyProfit->pluck('profit')->toArray();

        return view('analytics.profitability', compact(
            'productProfitability', 'categoryProfitability',
            'monthlyLabels', 'monthlyRevenue', 'monthlyCost', 'monthlyProfitData',
            'dateFrom', 'dateTo'
        ));
    }

    // Interactive Charts Dashboard
    public function charts(Request $request)
    {
        $dateFrom = $request->get('date_from', now()->subMonths(6)->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // Sales by day
        $dailySales = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->selectRaw('DATE(created_at) as date, SUM(total) as total, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Sales by payment method
        $paymentMethodSales = Payment::whereHas('invoice', function ($q) use ($dateFrom, $dateTo) {
                $q->whereBetween('created_at', [$dateFrom, $dateTo]);
            })
            ->selectRaw('payment_method, SUM(amount) as total')
            ->groupBy('payment_method')
            ->get();

        // Top products
        $topProducts = InvoiceItem::whereHas('invoice', function ($q) use ($dateFrom, $dateTo) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final'])
                  ->whereBetween('created_at', [$dateFrom, $dateTo]);
            })
            ->selectRaw('product_id, SUM(quantity) as total_quantity, SUM(total) as total_revenue')
            ->groupBy('product_id')
            ->orderByDesc('total_quantity')
            ->take(10)
            ->with('product')
            ->get();

        // Sales vs Purchases
        $salesData = Invoice::whereIn('status', ['paid', 'partially_paid', 'final'])
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->selectRaw('DATE(created_at) as date, SUM(total) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $purchasesData = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])
            ->whereBetween('purchase_date', [$dateFrom, $dateTo])
            ->selectRaw('DATE(purchase_date) as date, SUM(total_amount) as total')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        return view('analytics.charts', compact(
            'dailySales', 'paymentMethodSales', 'topProducts',
            'salesData', 'purchasesData', 'dateFrom', 'dateTo'
        ));
    }

    private function calculateProfit($invoices)
    {
        return $invoices->sum(function ($invoice) {
            return $invoice->items->sum(function ($item) {
                $revenue = $item->total;
                $cost = $item->quantity * ($item->product->purchase_price ?? 0);
                return $revenue - $cost;
            });
        });
    }
}
