<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Setting;
use Illuminate\Http\Request;
use App\Traits\Sortable;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\CategoriesImport;

class CategoryController extends Controller
{
    use Sortable;

    public function index(Request $request)
    {
        $query = Category::withCount('products');

        $allowedColumns = ['name', 'created_at', 'products_count'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            if ($sort === 'products_count') {
                $query->orderBy('products_count', $direction);
            } else {
                $query->orderBy($sort, $direction);
            }
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $categories = $query->paginate($itemsPerPage)->appends($request->query());
        return view('categories.index', compact('categories', 'sort', 'direction'));
    }

    public function create()
    {
        return view('categories.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
        ]);

        // If this is an AJAX request, return JSON response
        if ($request->expectsJson() || $request->ajax() || $request->routeIs('categories.store.ajax')) {
            try {
                $category = Category::create($validated);
                return response()->json([
                    'success' => true,
                    'message' => trans('messages.category_created'),
                    'category' => [
                        'id' => $category->id,
                        'name' => $category->name,
                        'name_ar' => $category->name_ar,
                    ]
                ]);
            } catch (\Illuminate\Validation\ValidationException $e) {
                return response()->json([
                    'success' => false,
                    'message' => trans('messages.error_occurred'),
                    'errors' => $e->errors()
                ], 422);
            } catch (\Exception $e) {
                return response()->json([
                    'success' => false,
                    'message' => trans('messages.error_occurred'),
                    'errors' => ['name' => [$e->getMessage()]]
                ], 422);
            }
        }

        Category::create($validated);

        return redirect()->route('categories.index')
            ->with('success', trans('messages.category_created'));
    }

    public function edit(Category $category)
    {
        return view('categories.edit', compact('category'));
    }

    public function update(Request $request, Category $category)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'name_ar' => 'nullable|string|max:255',
            'description' => 'nullable|string',
        ]);

        $category->update($validated);

        return redirect()->route('categories.index')
            ->with('success', trans('messages.category_updated'));
    }

    public function destroy(Category $category)
    {
        if ($category->products()->count() > 0) {
            return redirect()->route('categories.index')
                ->with('error', trans('messages.cannot_delete_category_with_products'));
        }

        $category->delete();

        return redirect()->route('categories.index')
            ->with('success', trans('messages.category_deleted'));
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Only Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:categories,id',
        ]);

        try {
            $selectedIds = $request->selected_ids;
            $categories = Category::whereIn('id', $selectedIds)->get();
            
            // Check for categories with products
            $categoriesWithProducts = $categories->filter(function ($category) {
                return $category->products()->count() > 0;
            });

            if ($categoriesWithProducts->count() > 0) {
                return back()->with('error', trans('messages.cannot_delete_category_with_products'));
            }
            
            // Log before deletion
            foreach ($categories as $category) {
                \App\Models\AuditLog::create([
                    'action' => 'bulk_delete',
                    'model_type' => Category::class,
                    'model_id' => $category->id,
                    'old_values' => $category->toArray(),
                    'new_values' => null,
                    'user_id' => auth()->id(),
                    'ip_address' => request()->ip(),
                    'user_agent' => request()->userAgent(),
                ]);
            }

            Category::whereIn('id', $selectedIds)->delete();

            return redirect()->route('categories.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function importForm()
    {
        // Check permissions - Only Admin and Super Admin can import
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        return view('categories.import');
    }

    public function import(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Admin', 'Super Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls|max:10240',
        ]);

        try {
            $file = $request->file('file');
            
            if (!$file->isValid()) {
                return back()->with('error', trans('messages.import_error') . ': ' . 'Invalid file.');
            }

            $import = new CategoriesImport();
            
            Excel::import($import, $file);
            
            $imported = $import->getImported();
            $failed = $import->getFailed();
            $errors = $import->getErrors();

            if ($imported === 0 && $failed === 0) {
                return back()->with('error', trans('messages.excel_file_empty'));
            }

            $message = trans('messages.import_completed', [
                'imported' => $imported,
                'failed' => $failed,
            ]);

            if ($failed > 0 && !empty($errors)) {
                return back()->with('warning', $message)->with('import_errors', $errors);
            }

            return back()->with('success', $message);

        } catch (\Maatwebsite\Excel\Validators\ValidationException $e) {
            $failures = $e->failures();
            $errors = [];
            
            foreach ($failures as $failure) {
                $errors[] = [
                    'row' => $failure->row(),
                    'name' => $failure->values()['name'] ?? 'N/A',
                    'error' => implode(', ', $failure->errors()),
                ];
            }
            
            return back()->with('error', trans('messages.import_error'))
                        ->with('import_errors', $errors);
                        
        } catch (\Maatwebsite\Excel\Exceptions\SheetNotFoundException $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Sheet not found in Excel file.');
        } catch (\PhpOffice\PhpSpreadsheet\Reader\Exception $e) {
            return back()->with('error', trans('messages.import_error') . ': ' . 'Error reading Excel file. Please check if the file is valid.');
        } catch (\Exception $e) {
            \Log::error('Category import error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            return back()->with('error', trans('messages.import_error') . ': ' . $e->getMessage());
        }
    }

    public function downloadTemplate()
    {
        $data = [
            ['name', 'name_ar', 'description'],
            ['Category 1', 'فئة 1', 'Category description 1'],
            ['Category 2', 'فئة 2', 'Category description 2'],
        ];

        $filename = 'categories_import_template_' . date('Y-m-d') . '.xlsx';
        
        $export = new class($data) implements \Maatwebsite\Excel\Concerns\FromArray {
            protected $data;
            
            public function __construct($data) {
                $this->data = $data;
            }
            
            public function array(): array {
                return $this->data;
            }
        };
        
        return Excel::download($export, $filename);
    }
}

