<?php

namespace App\Http\Controllers;

use App\Models\Document;
use App\Models\Customer;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class DocumentController extends Controller
{
    /**
     * Store a new document for customer or supplier
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'type' => 'required|in:customer,supplier',
                'id' => 'required|integer',
                'file' => 'required|file|max:10240', // Max 10MB
                'description' => 'nullable|string|max:1000',
            ]);

            // Get the model instance
            try {
                if ($validated['type'] === 'customer') {
                    $model = Customer::findOrFail($validated['id']);
                } else {
                    $model = Supplier::findOrFail($validated['id']);
                }
            } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
                return response()->json([
                    'success' => false,
                    'message' => trans('messages.' . ($validated['type'] === 'customer' ? 'customer_not_found' : 'supplier_not_found'))
                ], 404);
            }

            // Store the file
            $file = $request->file('file');
            $originalName = $file->getClientOriginalName();
            $fileName = Str::uuid() . '.' . $file->getClientOriginalExtension();
            $filePath = $file->storeAs(
                $validated['type'] . '-documents',
                $fileName,
                'public'
            );

            // Create document record
            $document = Document::create([
                'documentable_type' => get_class($model),
                'documentable_id' => $model->id,
                'name' => $originalName,
                'file_path' => $filePath,
                'file_name' => $fileName,
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'description' => $validated['description'] ?? null,
                'uploaded_by' => auth()->id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => trans('messages.document_uploaded_successfully'),
                'document' => $document->load('uploader')
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.validation_error'),
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            \Log::error('Document upload error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => trans('messages.error_uploading_document') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Download a document
     */
    public function download(Document $document)
    {
        if (!Storage::disk('public')->exists($document->file_path)) {
            return back()->with('error', trans('messages.document_not_found'));
        }

        return Storage::disk('public')->download(
            $document->file_path,
            $document->name
        );
    }

    /**
     * View a document (for images and PDFs)
     */
    public function view(Document $document)
    {
        if (!Storage::disk('public')->exists($document->file_path)) {
            abort(404, trans('messages.document_not_found'));
        }

        $file = Storage::disk('public')->get($document->file_path);
        $mimeType = Storage::disk('public')->mimeType($document->file_path);

        return response($file, 200)
            ->header('Content-Type', $mimeType)
            ->header('Content-Disposition', 'inline; filename="' . $document->name . '"');
    }

    /**
     * Delete a document
     */
    public function destroy(Document $document)
    {
        // Delete the file
        if (Storage::disk('public')->exists($document->file_path)) {
            Storage::disk('public')->delete($document->file_path);
        }

        // Delete the record
        $document->delete();

        return back()->with('success', trans('messages.document_deleted_successfully'));
    }

    /**
     * Get documents for a customer or supplier
     */
    public function index(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|in:customer,supplier',
            'id' => 'required|integer',
        ]);

        if ($validated['type'] === 'customer') {
            $model = Customer::findOrFail($validated['id']);
        } else {
            $model = Supplier::findOrFail($validated['id']);
        }

        $documents = $model->documents()
            ->with('uploader')
            ->latest()
            ->get();

        return response()->json($documents);
    }
}
