<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Models\Payment;
use App\Models\Treasury;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    public function store(Request $request, Invoice $invoice)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,bank_transfer,credit,cheque,promissory_note,online,other',
            'notes' => 'nullable|string',
        ]);

        if ($invoice->isDraft()) {
            return back()->with('error', trans('messages.cannot_add_payment_to_draft'));
        }

        if ($invoice->isReturned()) {
            return back()->with('error', trans('messages.cannot_add_payment_to_returned'));
        }

        $remainingDue = $invoice->due_amount;
        
        if ($validated['amount'] > $remainingDue) {
            return back()->with('error', trans('messages.payment_exceeds_remaining') . ': ' . number_format($remainingDue, 2));
        }

        DB::beginTransaction();
        try {
            $payment = Payment::create([
                'invoice_id' => $invoice->id,
                'amount' => $validated['amount'],
                'payment_method' => $validated['payment_method'],
                'notes' => $validated['notes'] ?? null,
                'user_id' => auth()->id(),
            ]);

            $invoice->paid_amount += $validated['amount'];
            $invoice->updateTotals();

            // Create treasury entry for payment
            Treasury::create([
                'date' => now()->toDateString(),
                'type' => 'income',
                'reference_type' => 'sale_invoice',
                'reference_id' => $invoice->id,
                'amount' => $validated['amount'],
                'description' => trans('messages.sales_payment_for_invoice') . ' #' . $invoice->invoice_number,
                'payment_method' => $validated['payment_method'] === 'bank_transfer' ? 'bank' : (in_array($validated['payment_method'], ['cheque', 'credit', 'online', 'promissory_note']) ? 'other' : $validated['payment_method']),
                'created_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('invoices.show', $invoice)
                ->with('success', trans('messages.payment_recorded'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_recording_payment') . ': ' . $e->getMessage());
        }
    }

    public function destroy(Payment $payment)
    {
        $invoice = $payment->invoice;

        if (!$invoice->isDraft()) {
            DB::beginTransaction();
            try {
                $invoice->paid_amount -= $payment->amount;
                $paymentAmount = $payment->amount;
                $paymentMethod = $payment->payment_method === 'bank_transfer' ? 'bank' : (in_array($payment->payment_method, ['cheque', 'credit', 'online', 'promissory_note']) ? 'other' : $payment->payment_method);
                $payment->delete();
                $invoice->updateTotals();

                // Delete treasury entry for payment
                Treasury::where('reference_type', 'sale_invoice')
                    ->where('reference_id', $invoice->id)
                    ->where('amount', $paymentAmount)
                    ->where('payment_method', $paymentMethod)
                    ->where('type', 'income')
                    ->orderBy('created_at', 'desc')
                    ->first()
                    ?->delete();

                DB::commit();

                return redirect()->route('invoices.show', $invoice)
                    ->with('success', trans('messages.payment_deleted'));
            } catch (\Exception $e) {
                DB::rollBack();
                return back()->with('error', trans('messages.error_deleting_payment') . ': ' . $e->getMessage());
            }
        }

        $payment->delete();
        return redirect()->route('invoices.show', $invoice)
            ->with('success', trans('messages.payment_deleted'));
    }
}

