<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;

class PermissionController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasRole('Super Admin')) {
                abort(403, trans('messages.unauthorized_action'));
            }
            return $next($request);
        });
    }

    public function index()
    {
        $users = User::with(['roles', 'permissions'])->paginate(20);
        $roles = Role::with('permissions')->get();
        $permissions = Permission::all()->groupBy(function ($permission) {
            $parts = explode('.', $permission->name);
            return $parts[0] ?? 'other';
        });

        return view('permissions.index', compact('users', 'roles', 'permissions'));
    }

    public function showUser(User $user)
    {
        $user->load(['roles', 'permissions']);
        $allRoles = Role::all();
        $allPermissions = Permission::all()->groupBy(function ($permission) {
            $parts = explode('.', $permission->name);
            return $parts[0] ?? 'other';
        });

        return view('permissions.user', compact('user', 'allRoles', 'allPermissions'));
    }

    public function updateUserRoles(Request $request, User $user)
    {
        $request->validate([
            'roles' => 'array',
            'roles.*' => 'exists:roles,id',
        ]);

        $user->syncRoles($request->roles ?? []);

        return redirect()->route('permissions.user', $user)
            ->with('success', trans('messages.user_roles_updated'));
    }

    public function updateUserPermissions(Request $request, User $user)
    {
        $request->validate([
            'permissions' => 'array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $user->syncPermissions($request->permissions ?? []);

        return redirect()->route('permissions.user', $user)
            ->with('success', trans('messages.user_permissions_updated'));
    }

    public function createRole(Request $request)
    {
        $request->validate([
            'name' => 'required|string|unique:roles,name',
        ], [
            'name.required' => trans('messages.role_name_required'),
            'name.unique' => trans('messages.role_name_exists'),
        ]);

        Role::create(['name' => $request->name, 'guard_name' => 'web']);

        return redirect()->route('permissions.index')
            ->with('success', trans('messages.role_created'));
    }

    public function updateRolePermissions(Request $request, Role $role)
    {
        $request->validate([
            'permissions' => 'array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        $role->syncPermissions($request->permissions ?? []);

        return redirect()->route('permissions.index')
            ->with('success', trans('messages.role_permissions_updated'));
    }

    public function destroyRole(Role $role)
    {
        // Prevent deletion of default roles
        $defaultRoles = ['Super Admin', 'Admin', 'Cashier', 'Warehouse'];
        if (in_array($role->name, $defaultRoles)) {
            return redirect()->route('permissions.index')
                ->with('error', trans('messages.cannot_delete_default_role'));
        }

        $role->delete();

        return redirect()->route('permissions.index')
            ->with('success', trans('messages.role_deleted'));
    }

    public function destroyPermission(Permission $permission)
    {
        $permission->delete();

        return redirect()->route('permissions.index')
            ->with('success', trans('messages.permission_deleted'));
    }

    public function createPermission(Request $request)
    {
        $request->validate([
            'name' => 'required|string|unique:permissions,name',
        ], [
            'name.required' => trans('messages.permission_name_required'),
            'name.unique' => trans('messages.permission_name_exists'),
        ]);

        Permission::create(['name' => $request->name, 'guard_name' => 'web']);

        return redirect()->route('permissions.index')
            ->with('success', trans('messages.permission_created'));
    }
}
