<?php

namespace App\Http\Controllers;

use App\Models\Category;
use App\Models\Product;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\Customer;
use App\Models\Payment;
use App\Models\Setting;
use App\Models\ProductBatch;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PosController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:Cashier|Admin|Super Admin');
    }

    public function index(Request $request)
    {
        $selectedCategory = $request->get('category_id');
        
        $categories = Category::all();
        
        // Build product query - show all products (not just in stock) for POS
        // Users can still see out-of-stock items but can't add them to cart
        $query = Product::with('category')->orderBy('name', 'asc');

        // Category filter
        if ($selectedCategory) {
            $query->where('category_id', $selectedCategory);
        }

        // Search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }

        // Get products with pagination - use POS specific setting
        $itemsPerPage = (int)Setting::get('pos_products_per_page', 20);
        $products = $query->paginate($itemsPerPage)->appends($request->query());
        
        $cart = session('pos_cart', []);
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $cartTotals = $this->calculateCartTotal($cart, $cartDiscount);

        // POS Settings
        $posSettings = [
            'show_images' => Setting::get('pos_show_images', '1') === '1',
            'show_stock' => Setting::get('pos_show_stock', '1') === '1',
            'enable_discount' => Setting::get('pos_enable_discount', '1') === '1',
            'enable_tax' => Setting::get('pos_enable_tax', '1') === '1',
            'auto_select_customer' => Setting::get('pos_auto_select_customer', '0') === '1',
            'auto_print' => Setting::get('pos_auto_print', '0') === '1',
            'enable_fullscreen' => Setting::get('pos_enable_fullscreen', '1') === '1',
            'enable_sound' => Setting::get('pos_enable_sound', '1') === '1',
            'enable_payment' => Setting::get('pos_enable_payment', '0') === '1',
        ];

        return view('pos.index', compact('categories', 'products', 'selectedCategory', 'cart', 'cartDiscount', 'cartTotals', 'posSettings'));
    }

    public function getProducts(Request $request)
    {
        $selectedCategory = $request->get('category_id');
        $search = $request->get('search', '');
        $page = $request->get('page', 1);
        $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
        $isRTL = app()->getLocale() === 'ar';
        
        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = StockManager::getDefaultWarehouse()?->id;
        }
        
        // Get POS settings
        $posSettings = [
            'show_images' => Setting::get('pos_show_images', '1') === '1',
            'show_stock' => Setting::get('pos_show_stock', '1') === '1',
        ];
        
        // Build product query
        $query = Product::with('category')->orderBy('name', 'asc');

        // Category filter
        if ($selectedCategory) {
            $query->where('category_id', $selectedCategory);
        }

        // Search filter
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }

        // Get products with pagination
        $itemsPerPage = (int)Setting::get('pos_products_per_page', 20);
        $products = $query->paginate($itemsPerPage, ['*'], 'page', $page);
        
        // Build HTML for products
        $productsHtml = view('pos.partials.products-grid', compact('products', 'posSettings', 'allowNegativeStock', 'warehouseId', 'isRTL'))->render();
        
        // Build pagination HTML
        $paginationHtml = '';
        if ($products->hasPages() && empty($search)) {
            $paginationHtml = $products->links()->toHtml();
        }
        
        return response()->json([
            'success' => true,
            'html' => $productsHtml,
            'pagination' => $paginationHtml,
            'count' => $products->count(),
            'total' => $products->total(),
            'current_page' => $products->currentPage(),
            'last_page' => $products->lastPage(),
        ]);
    }

    public function search(Request $request)
    {
        $selectedCategory = $request->get('category_id');
        $search = $request->get('search', '');
        $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
        $isRTL = app()->getLocale() === 'ar';
        
        // Build product query
        $query = Product::with('category')->orderBy('name', 'asc');

        // Category filter
        if ($selectedCategory) {
            $query->where('category_id', $selectedCategory);
        }

        // Search filter
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('name_ar', 'like', "%{$search}%")
                  ->orWhere('sku', 'like', "%{$search}%")
                  ->orWhere('barcode', 'like', "%{$search}%");
            });
        }

        // Get products with pagination (limit to 20 for faster response)
        $products = $query->limit(20)->get();
        
        // Build HTML for products
        $productsHtml = '';
        if ($products->isEmpty()) {
            $productsHtml = '<div class="col-12">
                <div class="alert alert-info text-center">
                    <i class="bi bi-info-circle"></i> ' . trans('messages.no_products_found') . '
                </div>
            </div>';
        } else {
            foreach ($products as $product) {
                $productName = ($isRTL && $product->name_ar) ? $product->name_ar : $product->name;
                $productImage = $product->image ? storage_url($product->image) : '';
                $stockBadge = '';
                $disabled = '';
                $maxQuantity = '';
                
                // Get warehouse ID if multiple warehouses mode
                $warehouseId = null;
                if (StockManager::isMultipleWarehouses()) {
                    $warehouseId = StockManager::getDefaultWarehouse()?->id;
                }
                
                $stockQuantity = StockManager::getStockQuantity($product, $warehouseId);
                
                if ($stockQuantity <= 0) {
                    $stockBadge = '<span class="badge bg-danger ms-2">' . trans('messages.out_of_stock') . '</span>';
                    if (!$allowNegativeStock) {
                        $disabled = 'disabled';
                    }
                } elseif ($product->isLowStock($warehouseId)) {
                    $stockBadge = '<span class="badge bg-warning ms-2">' . trans('messages.low_stock') . '</span>';
                }
                
                if (!$allowNegativeStock) {
                    $maxQuantity = 'max="' . max(1, $stockQuantity) . '"';
                }
                
                $productsHtml .= '<div class="col-md-4 col-sm-6">
                    <div class="card h-100 product-card ' . ($stockQuantity <= 0 ? 'opacity-75' : '') . '">
                        ' . ($productImage ? 
                            '<img src="' . $productImage . '" class="card-img-top" alt="' . htmlspecialchars($productName) . '" style="height: 150px; object-fit: cover;">' :
                            '<div class="card-img-top bg-light d-flex align-items-center justify-content-center" style="height: 150px;">
                                <i class="bi bi-basket" style="font-size: 3rem; color: #ccc;"></i>
                            </div>'
                        ) . '
                        <div class="card-body">
                            <h6 class="card-title">' . htmlspecialchars($productName) . '</h6>
                            <p class="card-text mb-2">
                                <strong>' . format_currency($product->selling_price) . '</strong>
                                <small class="text-muted d-block">
                                    ' . trans('messages.stock') . ': ' . $stockQuantity . '
                                    ' . $stockBadge . '
                                </small>
                            </p>
                            <div class="input-group mb-2">
                                <input type="number" class="form-control product-quantity" 
                                       data-product-id="' . $product->id . '" 
                                       value="1" min="1" ' . $maxQuantity . ' ' . $disabled . '>
                                <button class="btn btn-primary add-to-cart-btn ' . ($disabled ? 'disabled' : '') . '" 
                                        data-product-id="' . $product->id . '"
                                        data-product-name="' . htmlspecialchars($productName) . '"
                                        ' . $disabled . '>
                                    <i class="bi bi-cart-plus"></i> ' . trans('messages.add_to_cart') . '
                                </button>
                            </div>
                        </div>
                    </div>
                </div>';
            }
        }
        
        return response()->json([
            'success' => true,
            'html' => $productsHtml,
            'count' => $products->count()
        ]);
    }

    public function searchByBarcode(Request $request)
    {
        $barcode = $request->get('barcode', '');
        
        if (empty($barcode)) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.barcode_required')
            ], 400);
        }

        $product = Product::where('barcode', $barcode)->first();
        
        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.product_not_found')
            ], 404);
        }

        $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
        $isRTL = app()->getLocale() === 'ar';
        
        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = StockManager::getDefaultWarehouse()?->id;
        }
        
        $stockQuantity = StockManager::getStockQuantity($product, $warehouseId);
        
        // Check if product can be added to cart
        $canAdd = $allowNegativeStock || $stockQuantity > 0;
        
        return response()->json([
            'success' => true,
            'product' => [
                'id' => $product->id,
                'name' => $isRTL && $product->name_ar ? $product->name_ar : $product->name,
                'name_ar' => $product->name_ar,
                'selling_price' => $product->selling_price,
                'stock_quantity' => $stockQuantity,
                'can_add' => $canAdd,
                'is_low_stock' => $product->isLowStock($warehouseId),
                'is_out_of_stock' => $stockQuantity <= 0,
            ]
        ]);
    }

    public function addToCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $product = Product::findOrFail($request->product_id);

        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = StockManager::getDefaultWarehouse()?->id;
        }

        // Check stock availability (only if negative stock is not allowed)
        $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
        if (!$allowNegativeStock && !StockManager::checkStockAvailability($product, $request->quantity, $warehouseId)) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.insufficient_stock')
            ], 400);
        }

        $cart = session('pos_cart', []);
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $productId = $request->product_id;
        $quantity = $request->quantity;

        if (isset($cart[$productId])) {
            $newQuantity = $cart[$productId]['quantity'] + $quantity;
            // Check stock availability (only if negative stock is not allowed)
            $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
            if (!$allowNegativeStock && !StockManager::checkStockAvailability($product, $newQuantity, $warehouseId)) {
                return response()->json([
                    'success' => false,
                    'message' => trans('messages.insufficient_stock')
                ], 400);
            }
            $cart[$productId]['quantity'] = $newQuantity;
        } else {
            $cart[$productId] = [
                'product_id' => $productId,
                'name' => $product->name,
                'name_ar' => $product->name_ar,
                'quantity' => $quantity,
                'unit_price' => $product->selling_price,
                'tax_rate' => $product->tax_rate ?? 0,
                'discount' => 0,
                'discount_type' => 'fixed',
                'subtotal' => $product->selling_price * $quantity,
            ];
        }

        // Recalculate subtotal
        $this->recalculateCartItem($cart[$productId]);

        session(['pos_cart' => $cart, 'pos_cart_discount' => $cartDiscount]);

        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'message' => trans('messages.product_added_to_cart'),
            'cart' => $cart,
            'cart_total' => $totals['total'],
            'cart_subtotal' => $totals['subtotal'],
            'cart_tax' => $totals['tax'],
            'cart_discount' => $totals['discount'],
            'cart_count' => count($cart)
        ]);
    }

    public function updateCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'quantity' => 'required|integer|min:0',
        ]);

        $product = Product::findOrFail($request->product_id);
        $cart = session('pos_cart', []);
        $productId = $request->product_id;
        $quantity = $request->quantity;

        // Get warehouse ID if multiple warehouses mode
        $warehouseId = null;
        if (StockManager::isMultipleWarehouses()) {
            $warehouseId = StockManager::getDefaultWarehouse()?->id;
        }

        if ($quantity == 0) {
            unset($cart[$productId]);
        } else {
            // Check stock availability (only if negative stock is not allowed)
            $allowNegativeStock = \App\Models\Setting::get('allow_negative_stock', '0') === '1';
            if (!$allowNegativeStock && !StockManager::checkStockAvailability($product, $quantity, $warehouseId)) {
                return response()->json([
                    'success' => false,
                    'message' => trans('messages.insufficient_stock')
                ], 400);
            }

            if (isset($cart[$productId])) {
                $cart[$productId]['quantity'] = $quantity;
                $this->recalculateCartItem($cart[$productId]);
            }
        }

        session(['pos_cart' => $cart]);

        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'cart' => $cart,
            'cart_total' => $totals['total'],
            'cart_subtotal' => $totals['subtotal'],
            'cart_tax' => $totals['tax'],
            'cart_discount' => $totals['discount'],
            'cart_count' => count($cart)
        ]);
    }

    public function removeFromCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);

        $cart = session('pos_cart', []);
        $productId = $request->product_id;

        if (isset($cart[$productId])) {
            unset($cart[$productId]);
        }

        session(['pos_cart' => $cart]);

        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'message' => trans('messages.product_removed_from_cart'),
            'cart' => $cart,
            'cart_total' => $totals['total'],
            'cart_subtotal' => $totals['subtotal'],
            'cart_tax' => $totals['tax'],
            'cart_discount' => $totals['discount'],
            'cart_count' => count($cart)
        ]);
    }

    public function clearCart()
    {
        session([
            'pos_cart' => [],
            'pos_cart_discount' => ['type' => 'fixed', 'value' => 0]
        ]);

        return response()->json([
            'success' => true,
            'message' => trans('messages.cart_cleared'),
            'cart' => [],
            'cart_discount' => ['type' => 'fixed', 'value' => 0],
            'totals' => ['subtotal' => 0, 'tax' => 0, 'total' => 0, 'discount' => 0],
            'cart_count' => 0
        ]);
    }

    public function createInvoice(Request $request)
    {
        $cart = session('pos_cart', []);
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);

        if (empty($cart)) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.cart_is_empty')
            ], 400);
        }

        DB::beginTransaction();
        try {
            // Calculate totals
            $totals = $this->calculateCartTotal($cart, $cartDiscount);
            
            // Create draft invoice
            $invoice = Invoice::create([
                'invoice_number' => $this->generateInvoiceNumber(),
                'customer_id' => $request->customer_id ?? null,
                'customer_name' => $request->customer_name ?? trans('messages.walk_in_customer'),
                'status' => 'draft',
                'subtotal' => $totals['subtotal'],
                'tax' => $totals['tax'],
                'discount' => $totals['discount'],
                'total' => $totals['total'],
                'paid_amount' => 0,
                'due_amount' => $totals['total'],
                'user_id' => auth()->id(),
            ]);

            // Add items to invoice
            foreach ($cart as $item) {
                $product = Product::findOrFail($item['product_id']);
                
                $itemSubtotal = $item['unit_price'] * $item['quantity'];
                $itemDiscount = $item['discount'] ?? 0;
                $itemAfterDiscount = $itemSubtotal - $itemDiscount;
                $itemTax = $itemAfterDiscount * (($item['tax_rate'] ?? 0) / 100);
                $itemTotal = $itemAfterDiscount + $itemTax;

                InvoiceItem::create([
                    'invoice_id' => $invoice->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'discount' => $itemDiscount,
                    'subtotal' => $itemSubtotal,
                    'tax' => $itemTax,
                    'total' => $itemTotal,
                ]);
            }

            // Handle payment if provided
            $paidAmount = 0;
            $paymentMethod = null;
            $shouldUpdateStock = false;
            
            if ($request->has('payment_amount') && $request->payment_amount > 0) {
                $paidAmount = (float)$request->payment_amount;
                $paymentMethod = $request->payment_method ?? 'cash';
                
                // Create payment
                Payment::create([
                    'invoice_id' => $invoice->id,
                    'amount' => $paidAmount,
                    'payment_method' => $paymentMethod,
                    'payment_date' => now(),
                    'notes' => $request->payment_notes ?? 'Payment from POS',
                    'user_id' => auth()->id(),
                ]);
                
                // Update invoice
                $invoice->paid_amount = $paidAmount;
                $invoice->due_amount = $totals['total'] - $paidAmount;
                
                // Update status based on payment
                if ($paidAmount >= $totals['total']) {
                    $invoice->status = 'paid'; // Fully paid
                    $invoice->due_amount = 0;
                    $shouldUpdateStock = true; // Update stock when fully paid
                } elseif ($paidAmount > 0) {
                    $invoice->status = 'partially_paid'; // Partially paid
                    $shouldUpdateStock = true; // Update stock when partially paid (POS payment)
                } else {
                    $invoice->status = 'draft';
                }
            } else {
                // Invoice starts as draft - payments can be added later from invoice page
                $invoice->paid_amount = 0;
                $invoice->due_amount = $totals['total'];
                $invoice->status = 'draft';
            }
            
            $invoice->save();
            
            // Update stock if payment was made (POS payment)
            if ($shouldUpdateStock) {
                $allowNegativeStock = Setting::get('allow_negative_stock', '0') === '1';
                
                // Reload invoice with items to ensure they're loaded
                $invoice->load('items.product');
                
                foreach ($invoice->items as $item) {
                    $product = $item->product;
                    
                    // Get warehouse ID if multiple warehouses mode
                    $warehouseId = null;
                    if (StockManager::isMultipleWarehouses()) {
                        $warehouseId = StockManager::getDefaultWarehouse()?->id;
                    }
                    
                    // Check stock availability (only if negative stock is not allowed)
                    if (!$allowNegativeStock && !StockManager::checkStockAvailability($product, $item->quantity, $warehouseId)) {
                        DB::rollBack();
                        return response()->json([
                            'success' => false,
                            'message' => trans('messages.insufficient_stock') . ': ' . $product->name
                        ], 400);
                    }
                    
                    // Update batches using FIFO (First In First Out)
                    $remainingQuantity = $item->quantity;
                    $batches = \App\Models\ProductBatch::where('product_id', $product->id)
                        ->where('current_quantity', '>', 0)
                        ->orderBy('purchase_date', 'asc')
                        ->orderBy('id', 'asc')
                        ->get();
                    
                    $firstBatch = null;
                    foreach ($batches as $batch) {
                        if ($remainingQuantity <= 0) break;
                        
                        if ($firstBatch === null) {
                            $firstBatch = $batch; // Store first batch for invoice item
                        }
                        
                        $quantityToDeduct = min($remainingQuantity, $batch->current_quantity);
                        $batch->current_quantity -= $quantityToDeduct;
                        $batch->save();
                        $remainingQuantity -= $quantityToDeduct;
                    }
                    
                    // Link invoice item to first batch (for tracking)
                    if ($firstBatch) {
                        $item->batch_id = $firstBatch->id;
                        $item->save();
                    }
                    
                    // Update stock using StockManager (after batches to ensure consistency)
                    $notes = "Invoice #{$invoice->invoice_number} - POS Payment" . ($firstBatch ? " - Batch: {$firstBatch->batch_number}" : '');
                    StockManager::updateStock(
                        $product,
                        $item->quantity,
                        'OUT',
                        $warehouseId,
                        Invoice::class,
                        $invoice->id,
                        $notes
                    );
                }
                
                // Status is already set correctly above (paid or partially_paid)
                // No need to change it again
            }

            // Clear cart and discount
            session(['pos_cart' => [], 'pos_cart_discount' => ['type' => 'fixed', 'value' => 0]]);

            DB::commit();

            // Return JSON response with invoice info to open in new windows
            $autoPrint = Setting::get('pos_auto_print', '0') === '1';
            
            return response()->json([
                'success' => true,
                'message' => trans('messages.invoice_created_from_cart'),
                'invoice_id' => $invoice->id,
                'invoice_number' => $invoice->invoice_number,
                'show_url' => route('invoices.show', $invoice),
                'print_url' => route('invoices.print', $invoice),
                'auto_print' => $autoPrint,
                'paid_amount' => $paidAmount,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => trans('messages.error_creating_invoice') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    private function calculateCartTotal($cart, $cartDiscount = null)
    {
        $subtotal = 0;
        $tax = 0;
        
        foreach ($cart as $item) {
            $itemSubtotal = ($item['unit_price'] * $item['quantity']) - ($item['discount'] ?? 0);
            $itemTax = $itemSubtotal * (($item['tax_rate'] ?? 0) / 100);
            $subtotal += $itemSubtotal;
            $tax += $itemTax;
        }
        
        // Apply cart-level discount
        if ($cartDiscount) {
            $discountValue = $cartDiscount['value'] ?? 0;
            if ($cartDiscount['type'] === 'percentage') {
                $discountAmount = $subtotal * ($discountValue / 100);
            } else {
                $discountAmount = $discountValue;
            }
            $subtotal -= $discountAmount;
        }
        
        return [
            'subtotal' => $subtotal,
            'tax' => $tax,
            'total' => $subtotal + $tax,
            'discount' => $cartDiscount ? ($cartDiscount['type'] === 'percentage' ? $subtotal * (($cartDiscount['value'] ?? 0) / 100) : ($cartDiscount['value'] ?? 0)) : 0
        ];
    }
    
    private function recalculateCartItem(&$item)
    {
        $itemSubtotal = $item['unit_price'] * $item['quantity'];
        $discountValue = $item['discount'] ?? 0;
        $discountType = $item['discount_type'] ?? 'fixed';
        
        // Calculate discount amount based on type
        if ($discountType === 'percentage') {
            $itemDiscount = $itemSubtotal * ($discountValue / 100);
        } else {
            $itemDiscount = $discountValue;
        }
        
        $item['subtotal'] = $itemSubtotal - $itemDiscount;
    }
    
    public function updateCartDiscount(Request $request)
    {
        $request->validate([
            'discount_type' => 'required|in:fixed,percentage',
            'discount_value' => 'required|numeric|min:0',
        ]);
        
        $cart = session('pos_cart', []);
        $cartTotal = $this->calculateCartTotal($cart);
        
        // Validate discount value
        if ($request->discount_type === 'percentage' && $request->discount_value > 100) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.discount_percentage_invalid')
            ], 400);
        }
        
        if ($request->discount_type === 'fixed' && $request->discount_value > $cartTotal['subtotal']) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.discount_exceeds_total')
            ], 400);
        }
        
        $cartDiscount = [
            'type' => $request->discount_type,
            'value' => $request->discount_value,
        ];
        
        session(['pos_cart_discount' => $cartDiscount]);
        
        $newTotals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'cart' => $cart, // Include cart in response
            'cart_discount' => $cartDiscount,
            'totals' => $newTotals,
            'cart_count' => count($cart),
        ]);
    }
    
    public function updateItemDiscount(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'discount_type' => 'required|in:fixed,percentage',
            'discount_value' => 'required|numeric|min:0',
        ]);
        
        $cart = session('pos_cart', []);
        $productId = $request->product_id;
        
        if (!isset($cart[$productId])) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.product_not_in_cart')
            ], 404);
        }
        
        $item = $cart[$productId];
        $itemSubtotal = $item['unit_price'] * $item['quantity'];
        
        // Validate discount value
        if ($request->discount_type === 'percentage' && $request->discount_value > 100) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.discount_percentage_invalid')
            ], 400);
        }
        
        if ($request->discount_type === 'fixed' && $request->discount_value > $itemSubtotal) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.discount_exceeds_item_total')
            ], 400);
        }
        
        // Store discount value and type (calculation will be done in recalculateCartItem)
        $cart[$productId]['discount'] = $request->discount_value;
        $cart[$productId]['discount_type'] = $request->discount_type;
        
        $this->recalculateCartItem($cart[$productId]);
        
        session(['pos_cart' => $cart]);
        
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'item' => $cart[$productId],
            'cart' => $cart,
            'totals' => $totals,
            'cart_count' => count($cart),
        ]);
    }
    
    public function updateItemPrice(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
            'price' => 'required|numeric|min:0',
        ]);
        
        $cart = session('pos_cart', []);
        $productId = $request->product_id;
        
        if (!isset($cart[$productId])) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.product_not_in_cart')
            ], 404);
        }
        
        $cart[$productId]['unit_price'] = $request->price;
        $this->recalculateCartItem($cart[$productId]);
        
        session(['pos_cart' => $cart]);
        
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        return response()->json([
            'success' => true,
            'item' => $cart[$productId],
            'totals' => $totals,
        ]);
    }
    
    public function searchCustomers(Request $request)
    {
        $search = $request->get('search', '');
        
        $customers = Customer::where('name', 'like', "%{$search}%")
            ->orWhere('phone', 'like', "%{$search}%")
            ->orWhere('email', 'like', "%{$search}%")
            ->limit(10)
            ->get()
            ->map(function ($customer) {
                return [
                    'id' => $customer->id,
                    'name' => $customer->name,
                    'phone' => $customer->phone,
                    'email' => $customer->email,
                    'balance' => $customer->calculateCurrentBalance(),
                ];
            });
        
        return response()->json([
            'success' => true,
            'customers' => $customers,
        ]);
    }
    
    public function quickAddCustomer(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:50',
            'email' => 'nullable|email',
        ]);
        
        $customer = Customer::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
        ]);
        
        return response()->json([
            'success' => true,
            'customer' => [
                'id' => $customer->id,
                'name' => $customer->name,
                'phone' => $customer->phone,
                'email' => $customer->email,
                'balance' => 0,
            ],
        ]);
    }
    
    public function getTodayStats()
    {
        $today = now()->startOfDay();
        
        $invoices = Invoice::where('created_at', '>=', $today)
            ->whereIn('status', ['paid', 'partially_paid', 'final'])
            ->get();
        
        $totalSales = $invoices->sum('total');
        $invoiceCount = $invoices->count();
        $averageInvoice = $invoiceCount > 0 ? $totalSales / $invoiceCount : 0;
        
        return response()->json([
            'success' => true,
            'stats' => [
                'total_sales' => $totalSales,
                'invoice_count' => $invoiceCount,
                'average_invoice' => $averageInvoice,
            ],
        ]);
    }
    
    public function getRecentProducts()
    {
        $recentProducts = InvoiceItem::with('product')
            ->whereHas('invoice', function ($q) {
                $q->whereIn('status', ['paid', 'partially_paid', 'final'])
                  ->where('created_at', '>=', now()->subDays(7));
            })
            ->select('product_id', DB::raw('SUM(quantity) as total_quantity'))
            ->groupBy('product_id')
            ->orderByDesc('total_quantity')
            ->limit(10)
            ->get()
            ->map(function ($item) {
                if (!$item->product) return null;
                return [
                    'id' => $item->product->id,
                    'name' => $item->product->name,
                    'name_ar' => $item->product->name_ar,
                    'selling_price' => $item->product->selling_price,
                    'image' => $item->product->image,
                ];
            })
            ->filter();
        
        return response()->json([
            'success' => true,
            'products' => $recentProducts,
        ]);
    }
    
    public function getFavoriteProducts()
    {
        $favorites = session('pos_favorites', []);
        if (empty($favorites)) {
            return response()->json([
                'success' => true,
                'products' => [],
            ]);
        }
        
        $products = Product::whereIn('id', $favorites)->get();
        
        return response()->json([
            'success' => true,
            'products' => $products,
        ]);
    }
    
    public function toggleFavorite(Request $request)
    {
        $request->validate([
            'product_id' => 'required|exists:products,id',
        ]);
        
        $favorites = session('pos_favorites', []);
        $productId = $request->product_id;
        
        if (in_array($productId, $favorites)) {
            $favorites = array_values(array_diff($favorites, [$productId]));
            $isFavorite = false;
        } else {
            $favorites[] = $productId;
            $isFavorite = true;
        }
        
        session(['pos_favorites' => $favorites]);
        
        return response()->json([
            'success' => true,
            'is_favorite' => $isFavorite,
            'favorites' => $favorites,
        ]);
    }
    
    public function getProductBundles()
    {
        // For now, return empty - can be extended later with a bundles table
        return response()->json([
            'success' => true,
            'bundles' => [],
        ]);
    }
    
    public function saveDraft(Request $request)
    {
        $cart = session('pos_cart', []);
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $customerId = $request->get('customer_id');
        
        $draft = [
            'cart' => $cart,
            'cart_discount' => $cartDiscount,
            'customer_id' => $customerId,
            'saved_at' => now()->toDateTimeString(),
        ];
        
        $drafts = session('pos_drafts', []);
        $draftId = uniqid('draft_');
        $drafts[$draftId] = $draft;
        session(['pos_drafts' => $drafts]);
        
        return response()->json([
            'success' => true,
            'draft_id' => $draftId,
            'message' => trans('messages.draft_saved'),
        ]);
    }
    
    public function loadDraft(Request $request)
    {
        $draftId = $request->get('draft_id');
        $drafts = session('pos_drafts', []);
        
        if (!isset($drafts[$draftId])) {
            return response()->json([
                'success' => false,
                'message' => trans('messages.draft_not_found')
            ], 404);
        }
        
        $draft = $drafts[$draftId];
        session(['pos_cart' => $draft['cart']]);
        session(['pos_cart_discount' => $draft['cart_discount']]);
        
        return response()->json([
            'success' => true,
            'cart' => $draft['cart'],
            'cart_discount' => $draft['cart_discount'],
            'customer_id' => $draft['customer_id'] ?? null,
        ]);
    }
    
    public function getDrafts()
    {
        $drafts = session('pos_drafts', []);
        
        $draftsList = collect($drafts)->map(function ($draft, $id) {
            return [
                'id' => $id,
                'item_count' => count($draft['cart']),
                'total' => $this->calculateCartTotal($draft['cart'], $draft['cart_discount'])['total'],
                'saved_at' => $draft['saved_at'],
                'customer_id' => $draft['customer_id'] ?? null,
            ];
        })->values();
        
        return response()->json([
            'success' => true,
            'drafts' => $draftsList,
        ]);
    }

    private function generateInvoiceNumber()
    {
        // Get invoice number format from settings
        $format = \App\Models\Setting::get('invoice_number_format', 'INV-{YEAR}-{NUMBER}');
        
        // Replace {YEAR} with current year
        $year = date('Y');
        $formatWithYear = str_replace('{YEAR}', $year, $format);
        
        // Extract prefix from format (everything before {NUMBER})
        $prefix = str_replace('{NUMBER}', '', $formatWithYear);
        
        // Find all invoices that match the prefix pattern for current year
        $matchingInvoices = Invoice::where('invoice_number', 'like', $prefix . '%')
            ->where('invoice_number', 'not like', 'DEV-%') // Exclude quotations
            ->get();
        
        // Extract numbers from matching invoices
        $numbers = [];
        foreach ($matchingInvoices as $invoice) {
            // Extract numeric part from the end
            $invoiceNum = $invoice->invoice_number;
            // Remove prefix to get the number part
            $numberPart = str_replace($prefix, '', $invoiceNum);
            // Extract only numeric characters
            preg_match('/\d+/', $numberPart, $matches);
            if (!empty($matches)) {
                $numbers[] = (int)$matches[0];
            }
        }
        
        // Get the next number
        $number = empty($numbers) ? 1 : max($numbers) + 1;
        
        // Replace {NUMBER} with the generated number (padded to 6 digits)
        $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
        
        // Ensure uniqueness
        $counter = 0;
        while (Invoice::where('invoice_number', $invoiceNumber)->exists() && $counter < 1000) {
            $number++;
            $invoiceNumber = str_replace('{NUMBER}', str_pad($number, 6, '0', STR_PAD_LEFT), $formatWithYear);
            $counter++;
        }
        
        return $invoiceNumber;
    }

    public function printTicket(Request $request)
    {
        $cart = session('pos_cart', []);
        
        if (empty($cart)) {
            return redirect()->route('pos.index')
                ->with('error', trans('messages.cart_is_empty'));
        }

        $storeName = \App\Models\Setting::get('store_name', 'Gestock Pro');
        $storeLogo = \App\Models\Setting::get('store_logo');
        $currency = currency_symbol();
        $locale = app()->getLocale();
        $isRTL = $locale === 'ar';
        
        // Calculate totals with discounts
        $cartDiscount = session('pos_cart_discount', ['type' => 'fixed', 'value' => 0]);
        $totals = $this->calculateCartTotal($cart, $cartDiscount);
        
        $subtotal = $totals['subtotal'];
        $tax = $totals['tax'];
        $total = $totals['total'];
        $discount = $totals['discount'];
        
        // Get customer info
        $customerName = $request->get('customer_name', trans('messages.walk_in_customer'));
        $customerId = $request->get('customer_id');
        if ($customerId) {
            $customer = Customer::find($customerId);
            if ($customer) {
                $customerName = $customer->name;
            }
        }

        return view('pos.ticket', compact('cart', 'storeName', 'storeLogo', 'currency', 'locale', 'isRTL', 'subtotal', 'tax', 'total', 'discount', 'customerName'));
    }
}

