<?php

namespace App\Http\Controllers;

use App\Models\ProductDefect;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\PurchaseInvoice;
use App\Models\ProductBatch;
use App\Models\Setting;
use App\Models\User;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use App\Traits\Sortable;
use App\Traits\ChecksPermissions;

class ProductDefectController extends Controller
{
    use Sortable, ChecksPermissions;

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $this->checkPermission('quality.defects.view');
        
        $query = ProductDefect::with(['product', 'warehouse', 'reportedBy', 'assignedTo']);

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('defect_number', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%")
                  ->orWhereHas('product', function ($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%")
                        ->orWhere('name_ar', 'like', "%{$search}%");
                  });
            });
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('defect_type')) {
            $query->where('defect_type', $request->defect_type);
        }

        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('reported_date', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('reported_date', '<=', $request->date_to);
        }

        $allowedColumns = ['defect_number', 'reported_date', 'status', 'severity', 'created_at'];
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        if (in_array($sort, $allowedColumns)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->latest();
        }

        $itemsPerPage = Setting::getItemsPerPage(20);
        $defects = $query->paginate($itemsPerPage)->appends($request->query());
        $products = Product::all();
        $users = User::all();

        return view('quality.defects.index', compact('defects', 'products', 'users', 'sort', 'direction'));
    }

    public function create(Request $request)
    {
        $this->checkPermission('quality.defects.create');
        
        $products = Product::all();
        $warehouses = StockManager::isMultipleWarehouses() 
            ? Warehouse::active()->get() 
            : collect();
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])->get();
        $users = User::all();
        
        $productId = $request->get('product_id');
        $purchaseInvoiceId = $request->get('purchase_invoice_id');
        $batchId = $request->get('batch_id');

        return view('quality.defects.create', compact('products', 'warehouses', 'purchaseInvoices', 'users', 'productId', 'purchaseInvoiceId', 'batchId'));
    }

    public function store(Request $request)
    {
        $this->checkPermission('quality.defects.create');

        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'batch_id' => 'nullable|exists:product_batches,id',
            'defect_type' => 'required|in:manufacturing,transport,storage,other',
            'severity' => 'required|in:low,medium,high,critical',
            'quantity' => 'required|integer|min:1',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'status' => 'required|in:reported,under_review,resolved,rejected',
            'assigned_to' => 'nullable|exists:users,id',
            'reported_date' => 'required|date',
            'images' => 'nullable|array',
            'images.*' => 'image|max:2048',
        ]);

        DB::beginTransaction();
        try {
            $imagePaths = [];
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $imagePaths[] = $image->store('quality/defects', 'public');
                }
            }
            $validated['images'] = $imagePaths;
            $validated['reported_by'] = auth()->id();

            ProductDefect::create($validated);

            DB::commit();

            return redirect()->route('quality.defects.index')
                ->with('success', trans('messages.defect_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function show(ProductDefect $defect)
    {
        $this->checkPermission('quality.defects.view');
        
        $defect->load(['product', 'warehouse', 'purchaseInvoice', 'batch', 'reportedBy', 'assignedTo']);

        return view('quality.defects.show', compact('defect'));
    }

    public function edit(ProductDefect $defect)
    {
        $this->checkPermission('quality.defects.update');
        
        $products = Product::all();
        $warehouses = StockManager::isMultipleWarehouses() 
            ? Warehouse::active()->get() 
            : collect();
        $purchaseInvoices = PurchaseInvoice::whereIn('status', ['confirmed', 'paid', 'partially_paid'])->get();
        $users = User::all();

        return view('quality.defects.edit', compact('defect', 'products', 'warehouses', 'purchaseInvoices', 'users'));
    }

    public function update(Request $request, ProductDefect $defect)
    {
        $this->checkPermission('quality.defects.update');

        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'warehouse_id' => 'nullable|exists:warehouses,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'batch_id' => 'nullable|exists:product_batches,id',
            'defect_type' => 'required|in:manufacturing,transport,storage,other',
            'severity' => 'required|in:low,medium,high,critical',
            'quantity' => 'required|integer|min:1',
            'description' => 'required|string',
            'description_ar' => 'nullable|string',
            'status' => 'required|in:reported,under_review,resolved,rejected',
            'assigned_to' => 'nullable|exists:users,id',
            'reported_date' => 'required|date',
            'resolved_date' => 'nullable|date',
            'resolution_notes' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'image|max:2048',
        ]);

        DB::beginTransaction();
        try {
            // Handle image uploads
            $existingImages = $defect->images ?? [];
            $newImagePaths = [];
            
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $image) {
                    $newImagePaths[] = $image->store('quality/defects', 'public');
                }
            }

            // Keep existing images if not deleted
            if ($request->filled('existing_images')) {
                $existingImages = array_intersect($existingImages, $request->existing_images);
            } else {
                // Delete all existing images if none selected
                foreach ($existingImages as $image) {
                    Storage::disk('public')->delete($image);
                }
                $existingImages = [];
            }

            $validated['images'] = array_merge($existingImages, $newImagePaths);

            // Set resolved_date if status is resolved
            if ($validated['status'] === 'resolved' && !$defect->resolved_date) {
                $validated['resolved_date'] = now();
            }

            $defect->update($validated);

            DB::commit();

            return redirect()->route('quality.defects.show', $defect)
                ->with('success', trans('messages.defect_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withInput()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }

    public function destroy(ProductDefect $defect)
    {
        $this->checkPermission('quality.defects.delete');

        DB::beginTransaction();
        try {
            // Delete images
            if ($defect->images) {
                foreach ($defect->images as $image) {
                    Storage::disk('public')->delete($image);
                }
            }

            $defect->delete();

            DB::commit();

            return redirect()->route('quality.defects.index')
                ->with('success', trans('messages.defect_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_occurred') . ': ' . $e->getMessage());
        }
    }
}
