<?php

namespace App\Http\Controllers;

use App\Models\ProductDefect;
use App\Models\QualityComplaint;
use App\Models\Product;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class QualityReportController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        // Check permission
        if (!auth()->user()->can('quality.reports.view')) {
            abort(403, trans('messages.unauthorized_action'));
        }

        // Get date range
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->endOfMonth()->format('Y-m-d'));

        // Defects Statistics
        $totalDefects = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])->count();
        $resolvedDefects = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->where('status', 'resolved')->count();
        $criticalDefects = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->where('severity', 'critical')->count();
        
        $defectsByType = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->select('defect_type', DB::raw('count(*) as count'))
            ->groupBy('defect_type')
            ->get();
        
        $defectsBySeverity = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->select('severity', DB::raw('count(*) as count'))
            ->groupBy('severity')
            ->get();

        // Complaints Statistics
        $totalComplaints = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])->count();
        $resolvedComplaints = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->whereIn('status', ['resolved', 'closed'])->count();
        $criticalComplaints = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->where('severity', 'critical')->count();
        
        $complaintsByType = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->select('complaint_type', DB::raw('count(*) as count'))
            ->groupBy('complaint_type')
            ->get();
        
        $complaintsBySeverity = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->select('severity', DB::raw('count(*) as count'))
            ->groupBy('severity')
            ->get();

        // Products with most defects
        $productsWithDefects = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->select('product_id', DB::raw('count(*) as defect_count'))
            ->groupBy('product_id')
            ->orderByDesc('defect_count')
            ->limit(10)
            ->with('product')
            ->get();

        // Products with most complaints
        $productsWithComplaints = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->whereNotNull('product_id')
            ->select('product_id', DB::raw('count(*) as complaint_count'))
            ->groupBy('product_id')
            ->orderByDesc('complaint_count')
            ->limit(10)
            ->with('product')
            ->get();

        // Resolution time statistics
        $avgResolutionTime = ProductDefect::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->whereNotNull('resolved_date')
            ->selectRaw('AVG(DATEDIFF(resolved_date, reported_date)) as avg_days')
            ->first();

        $avgComplaintResolutionTime = QualityComplaint::whereBetween('reported_date', [$dateFrom, $dateTo])
            ->whereNotNull('resolved_date')
            ->selectRaw('AVG(DATEDIFF(resolved_date, reported_date)) as avg_days')
            ->first();

        return view('quality.reports.index', compact(
            'dateFrom',
            'dateTo',
            'totalDefects',
            'resolvedDefects',
            'criticalDefects',
            'defectsByType',
            'defectsBySeverity',
            'totalComplaints',
            'resolvedComplaints',
            'criticalComplaints',
            'complaintsByType',
            'complaintsBySeverity',
            'productsWithDefects',
            'productsWithComplaints',
            'avgResolutionTime',
            'avgComplaintResolutionTime'
        ));
    }
}
