<?php

namespace App\Http\Controllers;

use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;

class SettingsController extends Controller
{
    public function index()
    {
        // Check permissions: Only Super Admin and Admin can access settings
        if (!auth()->user()->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $settings = [
            // Store Settings
            'store_name' => Setting::get('store_name', 'Gestock Pro'),
            'store_logo' => Setting::get('store_logo'),
            'currency' => Setting::get('currency', '$'),
            'invoice_footer_text' => Setting::get('invoice_footer_text'),
            'items_per_page' => (int)Setting::get('items_per_page', 20),
            
            // POS Settings
            'pos_enabled' => Setting::get('pos_enabled', '1') === '1',
            'pos_show_images' => Setting::get('pos_show_images', '1') === '1',
            'pos_products_per_page' => (int)Setting::get('pos_products_per_page', 20),
            'pos_auto_select_customer' => Setting::get('pos_auto_select_customer', '0') === '1',
            'pos_show_stock' => Setting::get('pos_show_stock', '1') === '1',
            'pos_enable_discount' => Setting::get('pos_enable_discount', '1') === '1',
            'pos_enable_tax' => Setting::get('pos_enable_tax', '1') === '1',
            'pos_auto_print' => Setting::get('pos_auto_print', '0') === '1',
            'pos_enable_fullscreen' => Setting::get('pos_enable_fullscreen', '1') === '1',
            'pos_enable_sound' => Setting::get('pos_enable_sound', '1') === '1',
            'pos_enable_payment' => Setting::get('pos_enable_payment', '0') === '1',
            
            // Dashboard Settings
            'dashboard_show_sales_chart' => Setting::get('dashboard_show_sales_chart', '1') === '1',
            'dashboard_show_low_stock' => Setting::get('dashboard_show_low_stock', '1') === '1',
            'dashboard_show_recent_invoices' => Setting::get('dashboard_show_recent_invoices', '1') === '1',
            'dashboard_show_quality' => Setting::get('dashboard_show_quality', '1') === '1',
            'menu_show_quality' => Setting::get('menu_show_quality', '1') === '1',
            'dashboard_recent_invoices_count' => (int)Setting::get('dashboard_recent_invoices_count', 10),
            'dashboard_low_stock_count' => (int)Setting::get('dashboard_low_stock_count', 10),
            
            // Product Management Settings
            'sku_generation_mode' => Setting::get('sku_generation_mode', 'auto'),
            'sku_prefix' => Setting::get('sku_prefix', 'PROD-'),
            'barcode_format' => Setting::get('barcode_format', 'CODE128'),
            'show_product_name_in_barcode' => Setting::get('show_product_name_in_barcode', '0') === '1',
            'allow_negative_stock' => Setting::get('allow_negative_stock', '0') === '1',
            'warehouse_mode' => Setting::get('warehouse_mode', 'single'),
            'enable_batch_tracking' => Setting::get('enable_batch_tracking', '0') === '1',
            'auto_calculate_profit' => Setting::get('auto_calculate_profit', '1') === '1',
            
            // Invoice Settings
            'invoice_number_format' => Setting::get('invoice_number_format', 'INV-{YEAR}-{NUMBER}'),
            'invoice_show_logo' => Setting::get('invoice_show_logo', '1') === '1',
            'invoice_show_barcode' => Setting::get('invoice_show_barcode', '1') === '1',
            'invoice_default_status' => Setting::get('invoice_default_status', 'draft'),
            'invoice_auto_finalize' => Setting::get('invoice_auto_finalize', '0') === '1',
            'invoice_send_email' => Setting::get('invoice_send_email', '0') === '1',
            
            // Purchase Invoice Settings
            'purchase_invoice_number_format' => Setting::get('purchase_invoice_number_format', 'PINV-{YEAR}-{NUMBER}'),
            'purchase_invoice_show_logo' => Setting::get('purchase_invoice_show_logo', '1') === '1',
            'purchase_invoice_show_barcode' => Setting::get('purchase_invoice_show_barcode', '1') === '1',
            'purchase_invoice_default_status' => Setting::get('purchase_invoice_default_status', 'draft'),
            
            // Reports Settings
            'reports_default_date_range' => Setting::get('reports_default_date_range', 'month'),
            'reports_show_charts' => Setting::get('reports_show_charts', '1') === '1',
            'reports_export_format' => Setting::get('reports_export_format', 'excel'),
            
            // Backup Settings
            'auto_backup_enabled' => Setting::get('auto_backup_enabled', '0') === '1',
            'auto_backup_frequency' => Setting::get('auto_backup_frequency', 'daily'),
            'auto_backup_time' => Setting::get('auto_backup_time', '02:00'),
            'backup_retention_days' => (int)Setting::get('backup_retention_days', 14),
            
            // Notification Settings
            'notifications_enabled' => Setting::get('notifications_enabled', '1') === '1',
            'notifications_low_stock' => Setting::get('notifications_low_stock', '1') === '1',
            'notifications_pending_invoices' => Setting::get('notifications_pending_invoices', '1') === '1',
            
            // Activity Logs Settings
            'activity_logs_retention_days' => (int)Setting::get('activity_logs_retention_days', 30),
        ];

        return view('settings.index', compact('settings'));
    }

    public function update(Request $request)
    {
        // Check permissions: Only Super Admin and Admin can update settings
        if (!auth()->user()->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            // Store Settings
            'store_name' => 'required|string|max:255',
            'currency' => 'required|string|max:10',
            'invoice_footer_text' => 'nullable|string|max:500',
            'items_per_page' => 'required|integer|min:5|max:100',
            
            // POS Settings
            'pos_enabled' => 'nullable|boolean',
            'pos_show_images' => 'nullable|boolean',
            'pos_products_per_page' => 'required|integer|min:10|max:100',
            'pos_auto_select_customer' => 'nullable|boolean',
            'pos_show_stock' => 'nullable|boolean',
            'pos_enable_discount' => 'nullable|boolean',
            'pos_enable_tax' => 'nullable|boolean',
            'pos_auto_print' => 'nullable|boolean',
            'pos_enable_fullscreen' => 'nullable|boolean',
            'pos_enable_sound' => 'nullable|boolean',
            'pos_enable_payment' => 'nullable|boolean',
            
            // Dashboard Settings
            'dashboard_show_sales_chart' => 'nullable|boolean',
            'dashboard_show_low_stock' => 'nullable|boolean',
            'dashboard_show_recent_invoices' => 'nullable|boolean',
            'dashboard_show_quality' => 'nullable|boolean',
            'menu_show_quality' => 'nullable|boolean',
            'dashboard_recent_invoices_count' => 'required|integer|min:5|max:50',
            'dashboard_low_stock_count' => 'required|integer|min:5|max:50',
            
            // Product Management Settings
            'sku_generation_mode' => 'required|in:auto,manual',
            'sku_prefix' => 'nullable|string|max:50',
            'barcode_format' => 'required|in:CODE128,EAN13,EAN8,CODE39,ITF14,MSI,pharmacode,codabar',
            'show_product_name_in_barcode' => 'nullable|boolean',
            'allow_negative_stock' => 'nullable|boolean',
            'warehouse_mode' => 'required|in:single,multiple',
            'enable_batch_tracking' => 'nullable|boolean',
            'auto_calculate_profit' => 'nullable|boolean',
            
            // Invoice Settings
            'invoice_number_format' => 'nullable|string|max:100',
            'invoice_show_logo' => 'nullable|boolean',
            'invoice_show_barcode' => 'nullable|boolean',
            'invoice_default_status' => 'required|in:draft,final',
            'invoice_auto_finalize' => 'nullable|boolean',
            'invoice_send_email' => 'nullable|boolean',
            
            // Purchase Invoice Settings
            'purchase_invoice_number_format' => 'nullable|string|max:100',
            'purchase_invoice_show_logo' => 'nullable|boolean',
            'purchase_invoice_show_barcode' => 'nullable|boolean',
            'purchase_invoice_default_status' => 'required|in:draft,confirmed',
            
            // Reports Settings
            'reports_default_date_range' => 'required|in:today,week,month,quarter,year,custom',
            'reports_show_charts' => 'nullable|boolean',
            'reports_export_format' => 'required|in:excel,pdf',
            
            // Backup Settings
            'auto_backup_enabled' => 'nullable|boolean',
            'auto_backup_frequency' => 'required_if:auto_backup_enabled,1|nullable|in:daily,weekly',
            'auto_backup_time' => ['required_if:auto_backup_enabled,1', 'nullable', 'regex:/^([0-1][0-9]|2[0-3]):[0-5][0-9]$/'],
            'backup_retention_days' => 'required|integer|min:1|max:365',
            
            // Notification Settings
            'notifications_enabled' => 'nullable|boolean',
            'notifications_low_stock' => 'nullable|boolean',
            'notifications_pending_invoices' => 'nullable|boolean',
            
            // Activity Logs Settings
            'activity_logs_retention_days' => 'required|integer|min:0|max:365',
        ]);

        // Helper function to save setting
        $saveSetting = function($key, $value) {
            Setting::set($key, $value);
        };

        // Store Settings
        $saveSetting('store_name', $validated['store_name']);
        $saveSetting('currency', $validated['currency']);
        $saveSetting('invoice_footer_text', $validated['invoice_footer_text'] ?? '');
        $saveSetting('items_per_page', (string)$validated['items_per_page']);
        
        // POS Settings
        $saveSetting('pos_enabled', isset($validated['pos_enabled']) && $validated['pos_enabled'] ? '1' : '0');
        $saveSetting('pos_show_images', isset($validated['pos_show_images']) && $validated['pos_show_images'] ? '1' : '0');
        $saveSetting('pos_products_per_page', (string)$validated['pos_products_per_page']);
        $saveSetting('pos_auto_select_customer', isset($validated['pos_auto_select_customer']) && $validated['pos_auto_select_customer'] ? '1' : '0');
        $saveSetting('pos_show_stock', isset($validated['pos_show_stock']) && $validated['pos_show_stock'] ? '1' : '0');
        $saveSetting('pos_enable_discount', isset($validated['pos_enable_discount']) && $validated['pos_enable_discount'] ? '1' : '0');
        $saveSetting('pos_enable_tax', isset($validated['pos_enable_tax']) && $validated['pos_enable_tax'] ? '1' : '0');
        $saveSetting('pos_auto_print', isset($validated['pos_auto_print']) && $validated['pos_auto_print'] ? '1' : '0');
        $saveSetting('pos_enable_fullscreen', isset($validated['pos_enable_fullscreen']) && $validated['pos_enable_fullscreen'] ? '1' : '0');
        $saveSetting('pos_enable_sound', isset($validated['pos_enable_sound']) && $validated['pos_enable_sound'] ? '1' : '0');
        $saveSetting('pos_enable_payment', isset($validated['pos_enable_payment']) && $validated['pos_enable_payment'] ? '1' : '0');
        
        // Dashboard Settings
        $saveSetting('dashboard_show_sales_chart', isset($validated['dashboard_show_sales_chart']) && $validated['dashboard_show_sales_chart'] ? '1' : '0');
        $saveSetting('dashboard_show_low_stock', isset($validated['dashboard_show_low_stock']) && $validated['dashboard_show_low_stock'] ? '1' : '0');
        $saveSetting('dashboard_show_recent_invoices', isset($validated['dashboard_show_recent_invoices']) && $validated['dashboard_show_recent_invoices'] ? '1' : '0');
        $saveSetting('dashboard_show_quality', isset($validated['dashboard_show_quality']) && $validated['dashboard_show_quality'] ? '1' : '0');
        $saveSetting('menu_show_quality', isset($validated['menu_show_quality']) && $validated['menu_show_quality'] ? '1' : '0');
        $saveSetting('dashboard_recent_invoices_count', (string)$validated['dashboard_recent_invoices_count']);
        $saveSetting('dashboard_low_stock_count', (string)$validated['dashboard_low_stock_count']);
        
        // Product Management Settings
        $saveSetting('sku_generation_mode', $validated['sku_generation_mode']);
        $saveSetting('sku_prefix', $validated['sku_prefix'] ?? 'PROD-');
        $saveSetting('barcode_format', $validated['barcode_format']);
        $saveSetting('show_product_name_in_barcode', isset($validated['show_product_name_in_barcode']) && $validated['show_product_name_in_barcode'] ? '1' : '0');
        $saveSetting('allow_negative_stock', isset($validated['allow_negative_stock']) && $validated['allow_negative_stock'] ? '1' : '0');
        $saveSetting('warehouse_mode', $validated['warehouse_mode']);
        $saveSetting('enable_batch_tracking', isset($validated['enable_batch_tracking']) && $validated['enable_batch_tracking'] ? '1' : '0');
        $saveSetting('auto_calculate_profit', isset($validated['auto_calculate_profit']) && $validated['auto_calculate_profit'] ? '1' : '0');
        
        // Invoice Settings
        $saveSetting('invoice_number_format', $validated['invoice_number_format'] ?? 'INV-{YEAR}-{NUMBER}');
        $saveSetting('invoice_show_logo', isset($validated['invoice_show_logo']) && $validated['invoice_show_logo'] ? '1' : '0');
        $saveSetting('invoice_show_barcode', isset($validated['invoice_show_barcode']) && $validated['invoice_show_barcode'] ? '1' : '0');
        $saveSetting('invoice_default_status', $validated['invoice_default_status']);
        $saveSetting('invoice_auto_finalize', isset($validated['invoice_auto_finalize']) && $validated['invoice_auto_finalize'] ? '1' : '0');
        $saveSetting('invoice_send_email', isset($validated['invoice_send_email']) && $validated['invoice_send_email'] ? '1' : '0');
        
        // Purchase Invoice Settings
        $saveSetting('purchase_invoice_number_format', $validated['purchase_invoice_number_format'] ?? 'PINV-{YEAR}-{NUMBER}');
        $saveSetting('purchase_invoice_show_logo', isset($validated['purchase_invoice_show_logo']) && $validated['purchase_invoice_show_logo'] ? '1' : '0');
        $saveSetting('purchase_invoice_show_barcode', isset($validated['purchase_invoice_show_barcode']) && $validated['purchase_invoice_show_barcode'] ? '1' : '0');
        $saveSetting('purchase_invoice_default_status', $validated['purchase_invoice_default_status']);
        
        // Reports Settings
        $saveSetting('reports_default_date_range', $validated['reports_default_date_range']);
        $saveSetting('reports_show_charts', isset($validated['reports_show_charts']) && $validated['reports_show_charts'] ? '1' : '0');
        $saveSetting('reports_export_format', $validated['reports_export_format']);
        
        // Backup Settings
        $saveSetting('auto_backup_enabled', isset($validated['auto_backup_enabled']) && $validated['auto_backup_enabled'] ? '1' : '0');
        $saveSetting('auto_backup_frequency', $validated['auto_backup_frequency']);
        $saveSetting('auto_backup_time', $validated['auto_backup_time']);
        $saveSetting('backup_retention_days', (string)$validated['backup_retention_days']);
        
        // Notification Settings
        $saveSetting('notifications_enabled', isset($validated['notifications_enabled']) && $validated['notifications_enabled'] ? '1' : '0');
        $saveSetting('notifications_low_stock', isset($validated['notifications_low_stock']) && $validated['notifications_low_stock'] ? '1' : '0');
        $saveSetting('notifications_pending_invoices', isset($validated['notifications_pending_invoices']) && $validated['notifications_pending_invoices'] ? '1' : '0');
        
        // Activity Logs Settings
        $saveSetting('activity_logs_retention_days', (string)$validated['activity_logs_retention_days']);
        
        // Clear all settings cache
        Setting::clearCache();

        return redirect()->route('settings.index')
            ->with('success', trans('messages.settings_updated'));
    }

    public function uploadLogo(Request $request)
    {
        // Check permissions: Only Super Admin and Admin can upload logo
        if (!auth()->user()->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            'store_logo' => 'required|image|mimes:jpeg,jpg,png|max:2048',
        ]);

        // Delete old logo if exists
        $oldLogo = Setting::get('store_logo');
        if ($oldLogo && Storage::disk('public')->exists($oldLogo)) {
            Storage::disk('public')->delete($oldLogo);
        }

        // Store new logo
        $logoPath = $request->file('store_logo')->store('settings', 'public');
        Setting::set('store_logo', $logoPath);
        
        // Clear logo cache
        Cache::forget('setting.store_logo');

        return redirect()->route('settings.index')
            ->with('success', trans('messages.logo_uploaded'));
    }
}

