<?php

namespace App\Http\Controllers;

use App\Models\Treasury;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\TreasuryExport;

class TreasuryController extends Controller
{
    public function dashboard(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin', 'User'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $startDate = $request->get('start_date', date('Y-m-01'));
        $endDate = $request->get('end_date', date('Y-m-d'));
        $type = $request->get('type');
        $paymentMethod = $request->get('payment_method');

        $query = Treasury::query();

        if ($startDate && $endDate) {
            $query->dateRange($startDate, $endDate);
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($paymentMethod) {
            $query->byPaymentMethod($paymentMethod);
        }

        $totalIncome = (clone $query)->income()->sum('amount');
        $totalExpense = (clone $query)->expense()->sum('amount');
        $currentBalance = $totalIncome - $totalExpense;

        // Today's movements
        $todayIncome = Treasury::whereDate('date', today())->income()->sum('amount');
        $todayExpense = Treasury::whereDate('date', today())->expense()->sum('amount');
        $todayMovements = Treasury::whereDate('date', today())
            ->with('creator')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return view('treasury.dashboard', compact(
            'totalIncome',
            'totalExpense',
            'currentBalance',
            'todayIncome',
            'todayExpense',
            'todayMovements',
            'startDate',
            'endDate',
            'type',
            'paymentMethod'
        ));
    }

    public function index(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin', 'User'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $sort = $request->get('sort', 'date');
        $direction = $request->get('direction', 'desc');
        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        $type = $request->get('type');
        $paymentMethod = $request->get('payment_method');
        $referenceType = $request->get('reference_type');

        $query = Treasury::with('creator');

        // Apply filters
        if ($startDate && $endDate) {
            $query->dateRange($startDate, $endDate);
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($paymentMethod) {
            $query->byPaymentMethod($paymentMethod);
        }

        if ($referenceType) {
            $query->byReferenceType($referenceType);
        }

        // Apply sorting
        $allowedSorts = ['date', 'type', 'amount', 'payment_method', 'reference_type', 'created_at'];
        if (in_array($sort, $allowedSorts)) {
            $query->orderBy($sort, $direction);
        } else {
            $query->orderBy('date', 'desc');
        }

        $itemsPerPage = \App\Models\Setting::getItemsPerPage();
        $treasuries = $query->paginate($itemsPerPage)->withQueryString();

        return view('treasury.index', compact(
            'treasuries',
            'sort',
            'direction',
            'startDate',
            'endDate',
            'type',
            'paymentMethod',
            'referenceType'
        ));
    }

    public function create()
    {
        // Check permissions - only Admin and Super Admin can add manual entries
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        return view('treasury.create');
    }

    public function store(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            'date' => 'required|date',
            'type' => 'required|in:income,expense',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,card,bank,other',
            'description' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();
        try {
            Treasury::create([
                'date' => $validated['date'],
                'type' => $validated['type'],
                'reference_type' => 'manual',
                'reference_id' => null,
                'amount' => $validated['amount'],
                'description' => $validated['description'] ?? null,
                'payment_method' => $validated['payment_method'],
                'created_by' => auth()->id(),
            ]);

            DB::commit();

            return redirect()->route('treasury.index')
                ->with('success', trans('messages.treasury_entry_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_creating_treasury_entry') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function edit(Treasury $treasury)
    {
        // Check permissions - only Admin and Super Admin can edit
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        return view('treasury.edit', compact('treasury'));
    }

    public function update(Request $request, Treasury $treasury)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            'date' => 'required|date',
            'type' => 'required|in:income,expense',
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|in:cash,card,bank,other',
            'description' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();
        try {
            $treasury->update([
                'date' => $validated['date'],
                'type' => $validated['type'],
                'amount' => $validated['amount'],
                'description' => $validated['description'] ?? null,
                'payment_method' => $validated['payment_method'],
            ]);

            DB::commit();

            return redirect()->route('treasury.index')
                ->with('success', trans('messages.treasury_entry_updated'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_updating_treasury_entry') . ': ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Treasury $treasury)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin', 'User'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $treasury->load('creator');
        return view('treasury.show', compact('treasury'));
    }

    public function destroy(Treasury $treasury)
    {
        // Check permissions - Admin and Super Admin can delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        DB::beginTransaction();
        try {
            $treasury->delete();

            DB::commit();

            return redirect()->route('treasury.index')
                ->with('success', trans('messages.treasury_entry_deleted'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_deleting_treasury_entry') . ': ' . $e->getMessage());
        }
    }

    public function bulkDelete(Request $request)
    {
        // Check permissions - Admin and Super Admin can bulk delete
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $validated = $request->validate([
            'selected_ids' => 'required|array|min:1',
            'selected_ids.*' => 'exists:treasuries,id',
        ]);

        DB::beginTransaction();
        try {
            $selectedIds = $validated['selected_ids'];
            
            Treasury::whereIn('id', $selectedIds)->delete();

            DB::commit();

            return redirect()->route('treasury.index')
                ->with('success', trans('messages.bulk_delete_success', ['count' => count($selectedIds)]));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.bulk_delete_error') . ': ' . $e->getMessage());
        }
    }

    public function export(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin', 'User'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        $type = $request->get('type');
        $paymentMethod = $request->get('payment_method');
        $referenceType = $request->get('reference_type');

        $query = Treasury::with('creator');

        if ($startDate && $endDate) {
            $query->dateRange($startDate, $endDate);
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($paymentMethod) {
            $query->byPaymentMethod($paymentMethod);
        }

        if ($referenceType) {
            $query->byReferenceType($referenceType);
        }

        $treasuries = $query->orderBy('date', 'desc')->get();

        $filename = 'treasury_' . date('Y-m-d') . '.xlsx';
        return Excel::download(new TreasuryExport($treasuries), $filename);
    }

    public function exportPdf(Request $request)
    {
        // Check permissions
        $user = auth()->user();
        if (!$user->hasAnyRole(['Super Admin', 'Admin', 'User'])) {
            abort(403, trans('messages.unauthorized_action'));
        }

        $startDate = $request->get('start_date');
        $endDate = $request->get('end_date');
        $type = $request->get('type');
        $paymentMethod = $request->get('payment_method');
        $referenceType = $request->get('reference_type');

        $query = Treasury::with('creator');

        if ($startDate && $endDate) {
            $query->dateRange($startDate, $endDate);
        }

        if ($type) {
            $query->where('type', $type);
        }

        if ($paymentMethod) {
            $query->byPaymentMethod($paymentMethod);
        }

        if ($referenceType) {
            $query->byReferenceType($referenceType);
        }

        $treasuries = $query->orderBy('date', 'desc')->get();

        $totalIncome = (clone $query)->income()->sum('amount');
        $totalExpense = (clone $query)->expense()->sum('amount');
        $balance = $totalIncome - $totalExpense;

        // استخدام صفحة الطباعة لجميع اللغات
        return view('treasury.print', compact('treasuries', 'totalIncome', 'totalExpense', 'balance', 'startDate', 'endDate'));
    }

    /**
     * Get current treasury balance
     */
    public static function getCurrentBalance()
    {
        $totalIncome = Treasury::income()->sum('amount');
        $totalExpense = Treasury::expense()->sum('amount');
        return $totalIncome - $totalExpense;
    }
}
