<?php

namespace App\Http\Controllers;

use App\Models\Warehouse;
use App\Models\Product;
use App\Models\Setting;
use App\Services\StockManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class WarehouseTransferController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $warehouses = Warehouse::active()->get();
        $products = Product::all();

        return view('warehouses.transfer', compact('warehouses', 'products'));
    }

    public function store(Request $request)
    {
        if (Setting::get('warehouse_mode', 'single') !== 'multiple') {
            return redirect()->route('settings.index')
                ->with('error', trans('messages.multiple_warehouses_not_enabled'));
        }

        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'from_warehouse_id' => 'required|exists:warehouses,id',
            'to_warehouse_id' => 'required|exists:warehouses,id|different:from_warehouse_id',
            'quantity' => 'required|integer|min:1',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();
        try {
            $product = Product::findOrFail($validated['product_id']);

            StockManager::transferStock(
                $product,
                $validated['from_warehouse_id'],
                $validated['to_warehouse_id'],
                $validated['quantity'],
                $validated['notes'] ?? null
            );

            DB::commit();

            return redirect()->route('warehouses.transfer.index')
                ->with('success', trans('messages.transfer_created'));
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', trans('messages.error_transferring_stock') . ': ' . $e->getMessage())
                ->withInput();
        }
    }
}
