<?php

namespace App\Imports;

use App\Models\Category;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class CategoriesImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    public function collection(Collection $rows)
    {
        $categoriesToInsert = [];
        $rowNumber = 1;
        $usedNames = [];

        if ($rows->isEmpty()) {
            return;
        }

        $existingNames = Category::pluck('name')->toArray();

        foreach ($rows as $row) {
            $rowNumber++;
            
            try {
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                
                if (is_array($row)) {
                    $row = collect($row);
                }
                
                $name = $this->getColumnValue($row, ['name', 'nom', 'category_name', 'Name', 'NAME']);
                if (empty($name)) {
                    continue;
                }

                $name = trim($name);

                // Check for duplicate name (both in database and within batch)
                if (in_array($name, $existingNames) || in_array($name, $usedNames)) {
                    throw new \Exception(trans('messages.duplicate_name'));
                }
                $usedNames[] = $name;

                $categoryData = [
                    'name' => $name,
                    'name_ar' => $this->getColumnValue($row, ['name_ar', 'name_arabic', 'Name (AR)']) ? trim($this->getColumnValue($row, ['name_ar', 'name_arabic', 'Name (AR)'])) : null,
                    'description' => $this->getColumnValue($row, ['description', 'desc']) ? trim($this->getColumnValue($row, ['description', 'desc'])) : null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $categoriesToInsert[] = $categoryData;
                $this->imported++;

            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $name ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        if (!empty($categoriesToInsert)) {
            try {
                DB::beginTransaction();
                
                $chunks = array_chunk($categoriesToInsert, 500);
                foreach ($chunks as $chunk) {
                    Category::insert($chunk);
                }
                
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($categoriesToInsert);
                $this->imported -= count($categoriesToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Categories',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    public function getImported()
    {
        return $this->imported;
    }

    public function getFailed()
    {
        return $this->failed;
    }

    public function getErrors()
    {
        return $this->errors;
    }
}

