<?php

namespace App\Imports;

use App\Models\Customer;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class CustomersImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    public function collection(Collection $rows)
    {
        $customersToInsert = [];
        $rowNumber = 1;
        $usedEmails = [];

        if ($rows->isEmpty()) {
            return;
        }

        $existingEmails = Customer::whereNotNull('email')->pluck('email')->toArray();

        foreach ($rows as $row) {
            $rowNumber++;
            
            try {
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                
                if (is_array($row)) {
                    $row = collect($row);
                }
                
                $name = $this->getColumnValue($row, ['name', 'nom', 'customer_name', 'Name', 'NAME']);
                if (empty($name)) {
                    continue;
                }

                $name = trim($name);
                $email = $this->getColumnValue($row, ['email', 'Email', 'EMAIL']);
                $phone = $this->getColumnValue($row, ['phone', 'Phone', 'PHONE', 'telephone', 'tel']);
                $address = $this->getColumnValue($row, ['address', 'Address', 'ADDRESS', 'adresse']);
                $currentBalance = $this->parseNumeric($this->getColumnValue($row, ['current_balance', 'balance', 'Balance', 'CURRENT_BALANCE']));

                // Validate email if provided
                if (!empty($email)) {
                    $email = trim($email);
                    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        throw new \Exception(trans('messages.invalid_email'));
                    }
                    
                    // Check for duplicate email
                    if (in_array($email, $existingEmails) || in_array($email, $usedEmails)) {
                        throw new \Exception(trans('messages.duplicate_email'));
                    }
                    $usedEmails[] = $email;
                } else {
                    $email = null;
                }

                $customerData = [
                    'name' => $name,
                    'phone' => $phone ? trim($phone) : null,
                    'email' => $email,
                    'address' => $address ? trim($address) : null,
                    'current_balance' => $currentBalance >= 0 ? round($currentBalance, 2) : 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $customersToInsert[] = $customerData;
                $this->imported++;

            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $name ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        if (!empty($customersToInsert)) {
            try {
                DB::beginTransaction();
                
                $chunks = array_chunk($customersToInsert, 500);
                foreach ($chunks as $chunk) {
                    Customer::insert($chunk);
                }
                
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($customersToInsert);
                $this->imported -= count($customersToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Customers',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    private function parseNumeric($value)
    {
        if ($value === null || $value === '') {
            return 0;
        }
        
        if (is_numeric($value)) {
            return (float)$value;
        }
        
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$value);
        return !empty($cleaned) ? (float)$cleaned : 0;
    }

    public function getImported()
    {
        return $this->imported;
    }

    public function getFailed()
    {
        return $this->failed;
    }

    public function getErrors()
    {
        return $this->errors;
    }
}

