<?php

namespace App\Imports;

use App\Models\Invoice;
use App\Models\Customer;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class InvoicesImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    public function collection(Collection $rows)
    {
        $invoicesToInsert = [];
        $rowNumber = 1;
        $existingNumbers = Invoice::pluck('invoice_number')->toArray();
        $usedNumbers = [];

        if ($rows->isEmpty()) {
            return;
        }

        foreach ($rows as $row) {
            $rowNumber++;
            try {
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                if (is_array($row)) {
                    $row = collect($row);
                }

                $invoiceNumber = $this->getColumnValue($row, ['invoice_number', 'Invoice Number', 'INVOICE_NUMBER']);
                if (empty($invoiceNumber)) {
                    throw new \Exception(trans('messages.invoice_number_required'));
                }

                if (in_array($invoiceNumber, $existingNumbers) || in_array($invoiceNumber, $usedNumbers)) {
                    throw new \Exception(trans('messages.duplicate_invoice_number'));
                }
                $usedNumbers[] = $invoiceNumber;

                $customerId = $this->getColumnValue($row, ['customer_id', 'Customer ID', 'CUSTOMER_ID']);
                $customerName = $this->getColumnValue($row, ['customer_name', 'Customer Name', 'CUSTOMER_NAME']);
                
                if (empty($customerId) && empty($customerName)) {
                    throw new \Exception(trans('messages.customer_required'));
                }

                if (!empty($customerId)) {
                    $customer = Customer::find($customerId);
                    if (!$customer) {
                        throw new \Exception(trans('messages.customer_not_found'));
                    }
                    $customerName = $customer->name;
                }

                $status = $this->getColumnValue($row, ['status', 'Status', 'STATUS']) ?? 'draft';
                $validStatuses = ['draft', 'final', 'paid', 'partially_paid', 'returned'];
                if (!in_array($status, $validStatuses)) {
                    $status = 'draft';
                }

                $subtotal = $this->parseNumeric($this->getColumnValue($row, ['subtotal', 'Subtotal', 'SUBTOTAL'])) ?? 0;
                $tax = $this->parseNumeric($this->getColumnValue($row, ['tax', 'Tax', 'TAX'])) ?? 0;
                $discount = $this->parseNumeric($this->getColumnValue($row, ['discount', 'Discount', 'DISCOUNT'])) ?? 0;
                $total = $this->parseNumeric($this->getColumnValue($row, ['total', 'Total', 'TOTAL']));
                $paidAmount = $this->parseNumeric($this->getColumnValue($row, ['paid_amount', 'Paid Amount', 'PAID_AMOUNT'])) ?? 0;

                // Calculate total if not provided
                if ($total === null) {
                    $total = $subtotal + $tax - $discount;
                }

                // Calculate due amount
                $dueAmount = $total - $paidAmount;

                $invoicesToInsert[] = [
                    'invoice_number' => trim($invoiceNumber),
                    'customer_id' => $customerId ? (int)$customerId : null,
                    'customer_name' => trim($customerName),
                    'status' => $status,
                    'subtotal' => $subtotal,
                    'tax' => $tax,
                    'discount' => $discount,
                    'total' => $total,
                    'paid_amount' => $paidAmount,
                    'due_amount' => $dueAmount,
                    'notes' => $this->getColumnValue($row, ['notes', 'Notes', 'NOTES']) ? trim($this->getColumnValue($row, ['notes', 'Notes', 'NOTES'])) : null,
                    'user_id' => auth()->id(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
                $this->imported++;
            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $invoiceNumber ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        if (!empty($invoicesToInsert)) {
            try {
                DB::beginTransaction();
                $chunks = array_chunk($invoicesToInsert, 500);
                foreach ($chunks as $chunk) {
                    Invoice::insert($chunk);
                }
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($invoicesToInsert);
                $this->imported -= count($invoicesToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Invoices',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    private function parseNumeric($value)
    {
        if ($value === null || $value === '') {
            return null;
        }
        if (is_numeric($value)) {
            return (float)$value;
        }
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$value);
        return !empty($cleaned) ? (float)$cleaned : null;
    }

    public function getImported() { return $this->imported; }
    public function getFailed() { return $this->failed; }
    public function getErrors() { return $this->errors; }
}

