<?php

namespace App\Imports;

use App\Models\Product;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class ProductsImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    /**
     * Process the collection of rows
     *
     * @param Collection $rows
     */
    public function collection(Collection $rows)
    {
        $productsToInsert = [];
        $rowNumber = 1; // Start at 1 (header is row 1, first data row is 2)
        $usedSkus = []; // Track SKUs within this batch
        $usedBarcodes = []; // Track barcodes within this batch

        // Check if rows collection is empty
        if ($rows->isEmpty()) {
            return;
        }

        // Get existing SKUs and barcodes from database once
        $existingSkus = Product::pluck('sku')->toArray();
        $existingBarcodes = Product::whereNotNull('barcode')->pluck('barcode')->toArray();

        foreach ($rows as $row) {
            $rowNumber++;
            
            try {
                // Skip if row is not a collection or array
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                
                // Convert array to collection if needed
                if (is_array($row)) {
                    $row = collect($row);
                }
                
                // Skip empty rows
                $name = $this->getColumnValue($row, ['name', 'nom', 'product_name', 'Name', 'NAME']);
                if (empty($name)) {
                    continue;
                }

                // Read "Price" column as selling_price (case-insensitive matching)
                $sellingPrice = $this->parseNumeric($this->getColumnValue($row, ['price', 'Price', 'PRICE', 'selling_price', 'sell_price', 'Selling Price']));
                
                // Validate selling price (required)
                if ($sellingPrice <= 0) {
                    throw new \Exception(trans('messages.selling_price_invalid'));
                }

                // Read purchase_price from Excel (case-insensitive matching)
                $purchasePrice = $this->parseNumeric($this->getColumnValue($row, ['purchase_price', 'Purchase Price', 'PURCHASE_PRICE', 'cost', 'Cost', 'COST', 'prix_achat']));
                
                // Validate purchase price (required)
                if ($purchasePrice <= 0) {
                    throw new \Exception(trans('messages.purchase_price_invalid'));
                }

                // Read "qt_stock" column as stock_quantity (case-insensitive matching)
                $stockQuantity = (int)$this->parseNumeric($this->getColumnValue($row, ['qt_stock', 'Qt_Stock', 'QT_STOCK', 'quantity', 'Quantity', 'stock_quantity', 'Stock Quantity', 'qt_stoc']));
                // Default to 0 if empty or invalid
                if ($stockQuantity < 0) {
                    $stockQuantity = 0;
                }

                // Get SKU from row or generate one
                $sku = $this->getColumnValue($row, ['sku']);
                if (empty($sku)) {
                    $sku = $this->generateSku($name, $rowNumber);
                } else {
                    $sku = trim($sku);
                }

                // Check for duplicate SKU (both in database and within batch)
                while (in_array($sku, $existingSkus) || in_array($sku, $usedSkus)) {
                    $sku = $sku . '-' . $rowNumber;
                }
                $usedSkus[] = $sku;

                // Get barcode
                $barcode = $this->getColumnValue($row, ['barcode']);
                if (!empty($barcode)) {
                    $barcode = trim($barcode);
                    // Check for duplicate barcode (both in database and within batch)
                    if (in_array($barcode, $existingBarcodes) || in_array($barcode, $usedBarcodes)) {
                        $barcode = null; // Set to null if duplicate
                    } else {
                        $usedBarcodes[] = $barcode;
                    }
                }

                // Read brand (optional)
                $brand = $this->getColumnValue($row, ['brand', 'Brand', 'BRAND', 'marque', 'Marque']);
                $brand = $brand ? trim($brand) : null;

                // Prepare product data
                $productData = [
                    'name' => trim($name),
                    'name_ar' => $this->getColumnValue($row, ['name_ar', 'nom_ar', 'Name_ar']) ? trim($this->getColumnValue($row, ['name_ar', 'nom_ar', 'Name_ar'])) : null,
                    'brand' => $brand,
                    'barcode' => $barcode,
                    'sku' => $sku,
                    'purchase_price' => round($purchasePrice, 2),
                    'selling_price' => round($sellingPrice, 2),
                    'stock_quantity' => $stockQuantity,
                    'category_id' => null, // Always NULL as per requirements
                    'description' => $this->getColumnValue($row, ['description', 'desc']) ? trim($this->getColumnValue($row, ['description', 'desc'])) : null,
                    'unit' => 'pcs', // Default unit
                    'tax_rate' => 0,
                    'reorder_level' => 0,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $productsToInsert[] = $productData;
                $this->imported++;

            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $name ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        // Bulk insert all products in a single transaction
        if (!empty($productsToInsert)) {
            try {
                DB::beginTransaction();
                
                // Insert in chunks of 500 for better performance
                $chunks = array_chunk($productsToInsert, 500);
                foreach ($chunks as $chunk) {
                    Product::insert($chunk);
                }
                
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($productsToInsert);
                $this->imported -= count($productsToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Products',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    /**
     * Get column value from row, trying multiple possible column names
     */
    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            // Try exact match (case-insensitive)
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    // Handle null, empty string, and empty array
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    /**
     * Generate SKU if not provided
     */
    private function generateSku($name, $rowNumber)
    {
        return 'SKU-' . strtoupper(substr(md5($name . time() . $rowNumber), 0, 8));
    }

    /**
     * Parse numeric value from Excel cell
     */
    private function parseNumeric($value)
    {
        if ($value === null || $value === '') {
            return 0;
        }
        
        if (is_numeric($value)) {
            return (float)$value;
        }
        
        // Remove any non-numeric characters except decimal point
        $cleaned = preg_replace('/[^0-9.]/', '', (string)$value);
        return !empty($cleaned) ? (float)$cleaned : 0;
    }

    /**
     * Get import statistics
     */
    public function getImported()
    {
        return $this->imported;
    }

    /**
     * Get failed count
     */
    public function getFailed()
    {
        return $this->failed;
    }

    /**
     * Get errors
     */
    public function getErrors()
    {
        return $this->errors;
    }
}
