<?php

namespace App\Imports;

use App\Models\Supplier;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\SkipsOnFailure;
use Maatwebsite\Excel\Concerns\SkipsFailures;

class SuppliersImport implements ToCollection, WithHeadingRow, SkipsOnFailure
{
    use SkipsFailures;

    protected $imported = 0;
    protected $failed = 0;
    protected $errors = [];

    public function collection(Collection $rows)
    {
        $suppliersToInsert = [];
        $rowNumber = 1;
        $usedEmails = [];

        if ($rows->isEmpty()) {
            return;
        }

        $existingEmails = Supplier::whereNotNull('email')->pluck('email')->toArray();

        foreach ($rows as $row) {
            $rowNumber++;
            
            try {
                if (!($row instanceof Collection) && !is_array($row)) {
                    continue;
                }
                
                if (is_array($row)) {
                    $row = collect($row);
                }
                
                $name = $this->getColumnValue($row, ['name', 'nom', 'supplier_name', 'Name', 'NAME']);
                if (empty($name)) {
                    continue;
                }

                $name = trim($name);
                $email = $this->getColumnValue($row, ['email', 'Email', 'EMAIL']);
                $phone = $this->getColumnValue($row, ['phone', 'Phone', 'PHONE', 'telephone', 'tel']);
                $address = $this->getColumnValue($row, ['address', 'Address', 'ADDRESS', 'adresse']);
                $notes = $this->getColumnValue($row, ['notes', 'Notes', 'NOTES', 'note']);

                // Validate email if provided
                if (!empty($email)) {
                    $email = trim($email);
                    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                        throw new \Exception(trans('messages.invalid_email'));
                    }
                    
                    // Check for duplicate email
                    if (in_array($email, $existingEmails) || in_array($email, $usedEmails)) {
                        throw new \Exception(trans('messages.duplicate_email'));
                    }
                    $usedEmails[] = $email;
                } else {
                    $email = null;
                }

                $supplierData = [
                    'name' => $name,
                    'phone' => $phone ? trim($phone) : null,
                    'email' => $email,
                    'address' => $address ? trim($address) : null,
                    'notes' => $notes ? trim($notes) : null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];

                $suppliersToInsert[] = $supplierData;
                $this->imported++;

            } catch (\Exception $e) {
                $this->failed++;
                $this->errors[] = [
                    'row' => $rowNumber,
                    'name' => $name ?? 'N/A',
                    'error' => $e->getMessage(),
                ];
            }
        }

        if (!empty($suppliersToInsert)) {
            try {
                DB::beginTransaction();
                
                $chunks = array_chunk($suppliersToInsert, 500);
                foreach ($chunks as $chunk) {
                    Supplier::insert($chunk);
                }
                
                DB::commit();
            } catch (\Exception $e) {
                DB::rollBack();
                $this->failed += count($suppliersToInsert);
                $this->imported -= count($suppliersToInsert);
                $this->errors[] = [
                    'row' => 'Bulk Insert',
                    'name' => 'Multiple Suppliers',
                    'error' => 'Database error during bulk insert: ' . $e->getMessage(),
                ];
            }
        }
    }

    private function getColumnValue($row, array $possibleNames)
    {
        foreach ($possibleNames as $name) {
            $lowerName = strtolower(trim($name));
            foreach ($row->keys() as $key) {
                $lowerKey = strtolower(trim($key));
                if ($lowerKey === $lowerName) {
                    $value = $row->get($key);
                    if ($value !== null && $value !== '' && $value !== []) {
                        return is_string($value) ? trim($value) : $value;
                    }
                    return null;
                }
            }
        }
        return null;
    }

    public function getImported()
    {
        return $this->imported;
    }

    public function getFailed()
    {
        return $this->failed;
    }

    public function getErrors()
    {
        return $this->errors;
    }
}

