<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Customer extends Model
{
    use HasFactory;

    protected static function newFactory()
    {
        return \Database\Factories\CustomerFactory::new();
    }

    protected $fillable = [
        'name',
        'phone',
        'email',
        'address',
        'current_balance',
    ];

    protected function casts(): array
    {
        return [
            'current_balance' => 'decimal:2',
        ];
    }

    public function invoices()
    {
        return $this->hasMany(Invoice::class);
    }

    public function cheques()
    {
        return $this->hasMany(Cheque::class);
    }

    public function promissoryNotes()
    {
        return $this->hasMany(PromissoryNote::class);
    }

    public function documents()
    {
        return $this->morphMany(Document::class, 'documentable');
    }

    /**
     * Calculate current balance from unpaid and partially paid invoices
     * Current Balance = SUM(invoice.total) - SUM(invoice.paid_amount)
     * Only includes invoices with status: partially_paid, final (unpaid)
     * Excludes: paid, draft, returned
     */
    public function calculateCurrentBalance(): float
    {
        $unpaidInvoices = $this->invoices()
            ->whereIn('status', ['final', 'partially_paid'])
            ->get();

        $totalInvoiceAmount = $unpaidInvoices->sum('total');
        $totalPaidAmount = $unpaidInvoices->sum('paid_amount');

        $balance = $totalInvoiceAmount - $totalPaidAmount;
        return round($balance, 2);
    }
}

