<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Notification extends Model
{
    use HasFactory;

    protected $fillable = [
        'type',
        'title',
        'message',
        'severity',
        'icon',
        'is_read',
        'read_at',
        'model_type',
        'model_id',
        'data',
    ];

    protected $casts = [
        'is_read' => 'boolean',
        'read_at' => 'datetime',
        'data' => 'array',
    ];

    public function model()
    {
        return $this->morphTo();
    }

    public function markAsRead(): void
    {
        $this->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    public function markAsUnread(): void
    {
        $this->update([
            'is_read' => false,
            'read_at' => null,
        ]);
    }

    public function getSeverityBadgeAttribute(): string
    {
        return match($this->severity) {
            'danger' => 'danger',
            'warning' => 'warning',
            'success' => 'success',
            default => 'info',
        };
    }

    public function getUrlAttribute(): ?string
    {
        if (!$this->model_type || !$this->model_id) {
            return null;
        }

        $routeName = match($this->model_type) {
            Product::class => 'products.show',
            ProductBatch::class => 'products.show',
            Invoice::class => 'invoices.show',
            Cheque::class => 'cheques.show',
            PromissoryNote::class => 'promissory-notes.show',
            default => null,
        };

        if ($routeName) {
            try {
                return route($routeName, $this->model_id);
            } catch (\Exception $e) {
                return null;
            }
        }

        return null;
    }
}
