<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;

class Product extends Model
{
    use HasFactory;

    protected static function newFactory()
    {
        return \Database\Factories\ProductFactory::new();
    }

    protected $fillable = [
        'sku',
        'barcode',
        'name',
        'name_ar',
        'brand',
        'category_id',
        'unit',
        'purchase_price',
        'selling_price',
        'tax_rate',
        'stock_quantity',
        'reorder_level',
        'image',
        'description',
    ];

    protected function casts(): array
    {
        return [
            'purchase_price' => 'decimal:2',
            'selling_price' => 'decimal:2',
            'tax_rate' => 'decimal:2',
            'stock_quantity' => 'integer',
            'reorder_level' => 'integer',
        ];
    }

    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    public function invoiceItems()
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function stockMovements()
    {
        return $this->hasMany(StockMovement::class);
    }

    public function purchaseInvoiceItems()
    {
        return $this->hasMany(PurchaseInvoiceItem::class);
    }

    public function batches()
    {
        return $this->hasMany(ProductBatch::class);
    }

    public function warehouses()
    {
        return $this->belongsToMany(Warehouse::class, 'product_warehouse')
            ->withPivot('quantity', 'reorder_level')
            ->withTimestamps();
    }

    public function scopeLowStock(Builder $query)
    {
        $warehouseMode = Setting::get('warehouse_mode', 'single');
        
        if ($warehouseMode === 'multiple') {
            // For multiple warehouses, check if any warehouse has low stock
            return $query->whereHas('warehouses', function ($q) {
                $q->whereColumn('product_warehouse.quantity', '<=', 'product_warehouse.reorder_level');
            });
        }
        
        return $query->whereColumn('stock_quantity', '<=', 'reorder_level');
    }

    public function scopeInStock(Builder $query)
    {
        return $query->where('stock_quantity', '>', 0);
    }

    public function isLowStock($warehouseId = null): bool
    {
        $warehouseMode = Setting::get('warehouse_mode', 'single');
        
        if ($warehouseMode === 'multiple') {
            if ($warehouseId) {
                $pivot = $this->warehouses()->where('warehouses.id', $warehouseId)->first();
                if ($pivot) {
                    return $pivot->pivot->quantity <= $pivot->pivot->reorder_level;
                }
                return false;
            }
            // Check if any warehouse has low stock
            return $this->warehouses()->whereColumn('product_warehouse.quantity', '<=', 'product_warehouse.reorder_level')->exists();
        }
        
        return $this->stock_quantity <= $this->reorder_level;
    }

    public function getStockQuantity($warehouseId = null)
    {
        $warehouseMode = Setting::get('warehouse_mode', 'single');
        
        if ($warehouseMode === 'multiple') {
            if ($warehouseId) {
                $pivot = $this->warehouses()->where('warehouses.id', $warehouseId)->first();
                return $pivot ? $pivot->pivot->quantity : 0;
            }
            // Return total from all warehouses
            return $this->warehouses()->sum('product_warehouse.quantity');
        }
        
        return $this->stock_quantity;
    }

    public function getStockValueAttribute(): float
    {
        $warehouseMode = Setting::get('warehouse_mode', 'single');
        
        if ($warehouseMode === 'multiple') {
            $totalQuantity = $this->warehouses()->sum('product_warehouse.quantity');
            return $totalQuantity * $this->purchase_price;
        }
        
        return $this->stock_quantity * $this->purchase_price;
    }
}

