<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProductDefect extends Model
{
    use HasFactory;

    protected $fillable = [
        'product_id',
        'warehouse_id',
        'purchase_invoice_id',
        'batch_id',
        'defect_number',
        'defect_type',
        'severity',
        'quantity',
        'description',
        'description_ar',
        'status',
        'reported_by',
        'assigned_to',
        'reported_date',
        'resolved_date',
        'resolution_notes',
        'images',
    ];

    protected $casts = [
        'reported_date' => 'date',
        'resolved_date' => 'date',
        'images' => 'array',
        'quantity' => 'integer',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($defect) {
            if (empty($defect->defect_number)) {
                $lastDefect = static::where('defect_number', 'like', 'DEF-%')
                    ->latest()
                    ->first();
                $number = $lastDefect ? (int) substr($lastDefect->defect_number, -6) + 1 : 1;
                $defect->defect_number = 'DEF-' . str_pad($number, 6, '0', STR_PAD_LEFT);
            }
        });
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function warehouse(): BelongsTo
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function purchaseInvoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    public function batch(): BelongsTo
    {
        return $this->belongsTo(ProductBatch::class, 'batch_id');
    }

    public function reportedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reported_by');
    }

    public function assignedTo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    public function isResolved(): bool
    {
        return $this->status === 'resolved';
    }

    public function isUnderReview(): bool
    {
        return $this->status === 'under_review';
    }

    public function getSeverityBadgeClass(): string
    {
        return match($this->severity) {
            'critical' => 'danger',
            'high' => 'warning',
            'medium' => 'info',
            'low' => 'secondary',
            default => 'secondary',
        };
    }

    public function getStatusBadgeClass(): string
    {
        return match($this->status) {
            'resolved' => 'success',
            'under_review' => 'warning',
            'rejected' => 'danger',
            'reported' => 'info',
            default => 'secondary',
        };
    }
}
