<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class QualityComplaint extends Model
{
    use HasFactory;

    protected $fillable = [
        'complaint_number',
        'complaint_type',
        'product_id',
        'customer_id',
        'supplier_id',
        'invoice_id',
        'purchase_invoice_id',
        'complainant_name',
        'complainant_email',
        'complainant_phone',
        'description',
        'description_ar',
        'severity',
        'status',
        'reported_by',
        'assigned_to',
        'reported_date',
        'resolved_date',
        'resolution_notes',
        'images',
    ];

    protected $casts = [
        'reported_date' => 'date',
        'resolved_date' => 'date',
        'images' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($complaint) {
            if (empty($complaint->complaint_number)) {
                $lastComplaint = static::where('complaint_number', 'like', 'COMP-%')
                    ->latest()
                    ->first();
                $number = $lastComplaint ? (int) substr($lastComplaint->complaint_number, -6) + 1 : 1;
                $complaint->complaint_number = 'COMP-' . str_pad($number, 6, '0', STR_PAD_LEFT);
            }
        });
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function supplier(): BelongsTo
    {
        return $this->belongsTo(Supplier::class);
    }

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function purchaseInvoice(): BelongsTo
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    public function reportedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reported_by');
    }

    public function assignedTo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assigned_to');
    }

    public function isResolved(): bool
    {
        return $this->status === 'resolved' || $this->status === 'closed';
    }

    public function isInProgress(): bool
    {
        return $this->status === 'in_progress';
    }

    public function getSeverityBadgeClass(): string
    {
        return match($this->severity) {
            'critical' => 'danger',
            'high' => 'warning',
            'medium' => 'info',
            'low' => 'secondary',
            default => 'secondary',
        };
    }

    public function getStatusBadgeClass(): string
    {
        return match($this->status) {
            'resolved', 'closed' => 'success',
            'in_progress' => 'warning',
            'rejected' => 'danger',
            'new' => 'info',
            default => 'secondary',
        };
    }
}
