<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use App\Models\StockMovement;

class StockTake extends Model
{
    use HasFactory;

    protected $fillable = [
        'reference_number',
        'stock_take_date',
        'status',
        'type',
        'notes',
        'created_by',
        'completed_by',
        'completed_at',
    ];

    protected $casts = [
        'stock_take_date' => 'date',
        'completed_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($stockTake) {
            if (!$stockTake->reference_number) {
                $stockTake->reference_number = 'ST-' . strtoupper(Str::random(8));
            }
        });
    }

    public function items(): HasMany
    {
        return $this->hasMany(StockTakeItem::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function completer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'completed_by');
    }

    public function complete($userId): void
    {
        $this->update([
            'status' => 'completed',
            'completed_by' => $userId,
            'completed_at' => now(),
        ]);

        // Update product stock quantities based on variance
        foreach ($this->items as $item) {
            if ($item->variance != 0) {
                $product = $item->product;
                $product->stock_quantity = $item->counted_quantity;
                $product->save();

                // Log stock movement
                StockMovement::create([
                    'product_id' => $product->id,
                    'type' => $item->variance > 0 ? 'IN' : 'OUT',
                    'quantity' => abs($item->variance),
                    'quantity_before' => $item->system_quantity,
                    'quantity_after' => $item->counted_quantity,
                    'reference_type' => StockTake::class,
                    'reference_id' => $this->id,
                    'notes' => "Stock take adjustment - Reference: {$this->reference_number}",
                    'user_id' => $userId,
                ]);
            }
        }
    }

    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    public function isCompleted(): bool
    {
        return $this->status === 'completed';
    }
}
