<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Warehouse extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'name_ar',
        'code',
        'address',
        'phone',
        'is_default',
        'is_active',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'is_default' => 'boolean',
            'is_active' => 'boolean',
        ];
    }

    protected static function boot()
    {
        parent::boot();

        // Ensure only one default warehouse
        static::saving(function ($warehouse) {
            if ($warehouse->is_default) {
                static::where('id', '!=', $warehouse->id)
                    ->update(['is_default' => false]);
            }
        });
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_warehouse')
            ->withPivot('quantity', 'reorder_level')
            ->withTimestamps();
    }

    public function stockMovements()
    {
        return $this->hasMany(StockMovement::class);
    }

    public function scopeActive(Builder $query)
    {
        return $query->where('is_active', true);
    }

    public function scopeDefault(Builder $query)
    {
        return $query->where('is_default', true);
    }

    public static function getDefault()
    {
        return static::where('is_default', true)->first();
    }

    public function getTotalStockValueAttribute(): float
    {
        $totalValue = 0;
        foreach ($this->products as $product) {
            $pivot = $this->products()->where('products.id', $product->id)->first();
            if ($pivot) {
                $quantity = $pivot->pivot->quantity;
                $totalValue += $quantity * $product->purchase_price;
            }
        }
        return $totalValue;
    }

    public function getTotalProductsCountAttribute(): int
    {
        return $this->products()->where('product_warehouse.quantity', '>', 0)->count();
    }
}
