<?php

namespace App\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Config;
use App\Models\Setting;
use Barryvdh\DomPDF\Facade\Pdf;

class AppServiceProvider extends ServiceProvider
{
    public function register(): void
    {
        //
    }

    public function boot(): void
    {
        Schema::defaultStringLength(191);
        
        // Use Bootstrap 5 pagination
        \Illuminate\Pagination\Paginator::useBootstrapFive();
        
        // Auto-load settings on application boot
        try {
            // Load currency setting
            $currency = Setting::get('currency', '$');
            Config::set('app.currency', $currency);
            
            // Load store name
            $storeName = Setting::get('store_name', 'Gestock Pro');
            Config::set('app.store_name', $storeName);
            
            // Load store logo
            $storeLogo = Setting::get('store_logo');
            Config::set('app.store_logo', $storeLogo);
            
            // Load items per page
            $itemsPerPage = (int)Setting::get('items_per_page', 20);
            Config::set('app.items_per_page', $itemsPerPage);
            
            // Load invoice footer text
            $invoiceFooterText = Setting::get('invoice_footer_text', '');
            Config::set('app.invoice_footer_text', $invoiceFooterText);
        } catch (\Exception $e) {
            // Settings table might not exist during migrations
            // Silently fail and use defaults
        }
        
        // Register helper function for translating alert messages
        if (!function_exists('translate_alert_message')) {
            function translate_alert_message($message) {
                // Mapping of common English messages to translation keys
                $messageMap = [
                    'Invoice created successfully.' => 'invoice_created',
                    'Invoice updated successfully.' => 'invoice_updated',
                    'Invoice finalized successfully.' => 'invoice_finalized',
                    'Invoice returned successfully.' => 'invoice_returned',
                    'Product created successfully.' => 'product_created',
                    'Product updated successfully.' => 'product_updated',
                    'Product deleted successfully.' => 'product_deleted',
                    'Category created successfully.' => 'category_created',
                    'Category updated successfully.' => 'category_updated',
                    'Category deleted successfully.' => 'category_deleted',
                    'Customer created successfully.' => 'customer_created',
                    'Customer updated successfully.' => 'customer_updated',
                    'Customer deleted successfully.' => 'customer_deleted',
                    'Supplier created successfully.' => 'supplier_created',
                    'Supplier updated successfully.' => 'supplier_updated',
                    'Supplier deleted successfully.' => 'supplier_deleted',
                    'Payment recorded successfully.' => 'payment_recorded',
                    'Payment deleted successfully.' => 'payment_deleted',
                    'Only draft invoices can be edited.' => 'only_draft_can_be_edited',
                    'Only draft invoices can be finalized.' => 'only_draft_can_be_finalized',
                    'Invoice is already returned.' => 'invoice_already_returned',
                    'Draft invoices cannot be returned.' => 'draft_cannot_be_returned',
                    'Cannot delete category with associated products.' => 'cannot_delete_category_with_products',
                    'Cannot delete customer with associated invoices.' => 'cannot_delete_customers_with_invoices',
                    'Cannot add payment to draft invoice.' => 'cannot_add_payment_to_draft',
                    'Cannot add payment to returned invoice.' => 'cannot_add_payment_to_returned',
                    'Failed to create invoice: ' => 'error_creating_invoice',
                    'Failed to update invoice: ' => 'error_updating_invoice',
                    'Failed to finalize invoice: ' => 'error_finalizing_invoice',
                    'Failed to return invoice: ' => 'error_returning_invoice',
                    'Failed to record payment: ' => 'error_recording_payment',
                    'Failed to delete payment: ' => 'error_deleting_payment',
                ];
                
                // Check if message exists in map
                if (isset($messageMap[$message])) {
                    return trans('messages.' . $messageMap[$message]);
                }
                
                // Check if message starts with any mapped prefix
                foreach ($messageMap as $key => $transKey) {
                    if (strpos($message, $key) === 0) {
                        $remaining = substr($message, strlen($key));
                        return trans('messages.' . $transKey) . $remaining;
                    }
                }
                
                // Try direct translation as messages.key
                $translated = trans('messages.' . $message);
                if ($translated !== 'messages.' . $message) {
                    return $translated;
                }
                
                // Try as direct key
                $directTrans = trans($message);
                if ($directTrans !== $message) {
                    return $directTrans;
                }
                
                // Return original message if no translation found
                return $message;
            }
        }
    }
}
