<?php

namespace App\Services;

use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Crypt;
use Exception;

class LicenseDatabaseService
{
    protected $dbPath;
    protected $connectionName = 'license';
    protected $dbPassword;

    public function __construct()
    {
        // مسار قاعدة بيانات SQLite للتراخيص
        $this->dbPath = storage_path('app/licenses.db');
        
        // كلمة مرور قاعدة البيانات (من .env أو قيمة افتراضية)
        $this->dbPassword = env('LICENSE_DB_PASSWORD', 'GestionStock_License_2024_Secure_Password_Change_This');
        
        // إنشاء الاتصال إذا لم يكن موجوداً
        $this->setupConnection();
    }

    /**
     * إعداد اتصال قاعدة بيانات SQLite منفصلة
     */
    protected function setupConnection(): void
    {
        // إنشاء مجلد إذا لم يكن موجوداً
        $dir = dirname($this->dbPath);
        if (!File::exists($dir)) {
            File::makeDirectory($dir, 0755, true);
        }

        // إنشاء ملف قاعدة البيانات إذا لم يكن موجوداً
        if (!File::exists($this->dbPath)) {
            File::put($this->dbPath, '');
        }

        // إضافة الاتصال إلى config
        // ملاحظة: SQLite العادي لا يدعم كلمات المرور مباشرة
        // لكننا نستخدم تشفير البيانات + حماية الملف بصلاحيات النظام
        config([
            "database.connections.{$this->connectionName}" => [
                'driver' => 'sqlite',
                'database' => $this->dbPath,
                'prefix' => '',
                'foreign_key_constraints' => true,
                'password' => $this->dbPassword, // محفوظ للتوثيق (لا يستخدم في SQLite العادي)
            ],
        ]);
        
        // حماية الملف بصلاحيات النظام (على Linux/Mac)
        if (file_exists($this->dbPath) && PHP_OS_FAMILY !== 'Windows') {
            @chmod($this->dbPath, 0600); // قراءة وكتابة للمالك فقط
        }

        // إنشاء الجدول إذا لم يكن موجوداً
        $this->createTableIfNotExists();
    }

    /**
     * إنشاء جدول التراخيص في قاعدة البيانات المنفصلة
     */
    protected function createTableIfNotExists(): void
    {
        try {
            DB::connection($this->connectionName)->statement("
                CREATE TABLE IF NOT EXISTS licenses (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    license_key TEXT NOT NULL,
                    license_key_hash TEXT NOT NULL,
                    serial_number TEXT NOT NULL,
                    serial_number_hash TEXT NOT NULL,
                    domain TEXT,
                    machine_id TEXT,
                    machine_name TEXT,
                    ip_address TEXT,
                    email TEXT,
                    customer_name TEXT,
                    status TEXT DEFAULT 'pending',
                    activated_at TEXT,
                    expires_at TEXT,
                    max_users INTEGER DEFAULT 10,
                    notes TEXT,
                    created_at TEXT DEFAULT CURRENT_TIMESTAMP,
                    updated_at TEXT DEFAULT CURRENT_TIMESTAMP
                )
            ");

            // إنشاء فهارس
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_license_key_hash ON licenses(license_key_hash)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_serial_number_hash ON licenses(serial_number_hash)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_machine_id ON licenses(machine_id)
            ");
            DB::connection($this->connectionName)->statement("
                CREATE INDEX IF NOT EXISTS idx_status ON licenses(status)
            ");
            
            // إضافة الأعمدة الجديدة إذا لم تكن موجودة (للتوافق مع الجداول القديمة)
            try {
                DB::connection($this->connectionName)->statement("
                    ALTER TABLE licenses ADD COLUMN license_key_hash TEXT
                ");
            } catch (Exception $e) {
                // العمود موجود بالفعل
            }
            
            try {
                DB::connection($this->connectionName)->statement("
                    ALTER TABLE licenses ADD COLUMN serial_number_hash TEXT
                ");
            } catch (Exception $e) {
                // العمود موجود بالفعل
            }
        } catch (Exception $e) {
            \Log::error('Error creating license database: ' . $e->getMessage());
        }
    }

    /**
     * الحصول على اتصال قاعدة البيانات
     */
    public function getConnection()
    {
        return DB::connection($this->connectionName);
    }

    /**
     * إدراج ترخيص جديد
     */
    public function insert(array $data): bool
    {
        try {
            // إنشاء hash للبحث السريع (قبل التشفير)
            $licenseKeyHash = hash('sha256', strtoupper($data['license_key'] ?? ''));
            $serialNumberHash = hash('sha256', strtoupper($data['serial_number'] ?? ''));
            
            // تشفير البيانات الحساسة
            $encryptedData = $this->encryptSensitiveData($data);
            
            // إضافة hash للبحث السريع (غير مشفر)
            $encryptedData['license_key_hash'] = $licenseKeyHash;
            $encryptedData['serial_number_hash'] = $serialNumberHash;
            
            // إضافة timestamps
            $encryptedData['created_at'] = now()->toDateTimeString();
            $encryptedData['updated_at'] = now()->toDateTimeString();
            
            $result = $this->getConnection()->table('licenses')->insert($encryptedData);
            
            if (!$result) {
                \Log::error('Failed to insert license - insert returned false');
                return false;
            }
            
            return true;
        } catch (\PDOException $e) {
            \Log::error('PDO Error inserting license: ' . $e->getMessage());
            \Log::error('Error code: ' . $e->getCode());
            return false;
        } catch (Exception $e) {
            \Log::error('Error inserting license: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return false;
        }
    }

    /**
     * البحث عن ترخيص باستخدام hash للبحث السريع
     */
    public function find(string $licenseKey, ?string $serialNumber = null): ?array
    {
        try {
            $licenseKeyHash = hash('sha256', strtoupper($licenseKey));
            
            $query = $this->getConnection()->table('licenses')
                ->where('license_key_hash', $licenseKeyHash);
            
            if ($serialNumber) {
                $serialNumberHash = hash('sha256', strtoupper($serialNumber));
                $query->where('serial_number_hash', $serialNumberHash);
            }
            
            $license = $query->first();
            
            if (!$license) {
                return null;
            }
            
            // فك التشفير
            return $this->decryptSensitiveData((array)$license);
        } catch (Exception $e) {
            \Log::error('Error finding license: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * تحديث ترخيص
     */
    public function update(string $licenseKey, array $data): bool
    {
        try {
            $licenseKeyHash = hash('sha256', strtoupper($licenseKey));
            
            // البحث عن الترخيص أولاً
            $license = $this->getConnection()->table('licenses')
                ->where('license_key_hash', $licenseKeyHash)
                ->first();
            
            if (!$license) {
                return false;
            }
            
            // فك تشفير البيانات الحالية
            $currentData = $this->decryptSensitiveData((array)$license);
            
            // دمج البيانات الحالية مع الجديدة
            $mergedData = array_merge($currentData, $data);
            
            // تشفير البيانات الحساسة
            $encryptedData = $this->encryptSensitiveData($mergedData);
            
            // تحديث hash إذا تغير license_key أو serial_number
            if (isset($data['license_key'])) {
                $encryptedData['license_key_hash'] = hash('sha256', strtoupper($data['license_key']));
            }
            if (isset($data['serial_number'])) {
                $encryptedData['serial_number_hash'] = hash('sha256', strtoupper($data['serial_number']));
            }
            
            $encryptedData['updated_at'] = now()->toDateTimeString();
            
            // تحديث
            $this->getConnection()->table('licenses')
                ->where('id', $license->id)
                ->update($encryptedData);
            
            return true;
        } catch (Exception $e) {
            \Log::error('Error updating license: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * الحصول على جميع التراخيص
     */
    public function all(): array
    {
        try {
            $licenses = $this->getConnection()->table('licenses')->get();
            
            return $licenses->map(function ($license) {
                return $this->decryptSensitiveData((array)$license);
            })->toArray();
        } catch (Exception $e) {
            \Log::error('Error getting all licenses: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * البحث عن ترخيص حسب Machine ID
     */
    public function findByMachineId(string $machineId): ?array
    {
        try {
            $license = $this->getConnection()->table('licenses')
                ->where('machine_id', $machineId)
                ->where('status', 'active')
                ->first();
            
            if (!$license) {
                return null;
            }
            
            return $this->decryptSensitiveData((array)$license);
        } catch (Exception $e) {
            \Log::error('Error finding license by machine ID: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * تشفير البيانات الحساسة
     */
    protected function encryptSensitiveData(array $data): array
    {
        $sensitiveFields = ['license_key', 'serial_number', 'email', 'customer_name', 'notes'];
        
        foreach ($sensitiveFields as $field) {
            if (isset($data[$field]) && $data[$field] !== null) {
                $data[$field] = Crypt::encryptString($data[$field]);
            }
        }
        
        return $data;
    }

    /**
     * فك تشفير البيانات الحساسة
     */
    protected function decryptSensitiveData(array $data): array
    {
        $sensitiveFields = ['license_key', 'serial_number', 'email', 'customer_name', 'notes'];
        
        foreach ($sensitiveFields as $field) {
            if (isset($data[$field]) && $data[$field] !== null) {
                try {
                    $data[$field] = Crypt::decryptString($data[$field]);
                } catch (Exception $e) {
                    // إذا فشل فك التشفير، قد يكون غير مشفر (للتوافق مع البيانات القديمة)
                    // نتركه كما هو
                }
            }
        }
        
        return $data;
    }

    /**
     * نسخ احتياطي لقاعدة البيانات
     */
    public function backup(string $backupPath = null): bool
    {
        try {
            if (!$backupPath) {
                $backupPath = storage_path('app/backups/licenses_' . date('Y-m-d_His') . '.db');
            }
            
            $backupDir = dirname($backupPath);
            if (!File::exists($backupDir)) {
                File::makeDirectory($backupDir, 0755, true);
            }
            
            return File::copy($this->dbPath, $backupPath);
        } catch (Exception $e) {
            \Log::error('Error backing up license database: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * استعادة من نسخة احتياطية
     */
    public function restore(string $backupPath): bool
    {
        try {
            if (!File::exists($backupPath)) {
                return false;
            }
            
            // إغلاق الاتصال الحالي
            DB::disconnect($this->connectionName);
            
            // نسخ الملف
            File::copy($backupPath, $this->dbPath);
            
            // إعادة الاتصال
            $this->setupConnection();
            
            return true;
        } catch (Exception $e) {
            \Log::error('Error restoring license database: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * الحصول على معلومات قاعدة البيانات
     */
    public function getDatabaseInfo(): array
    {
        $info = [
            'path' => $this->dbPath,
            'exists' => File::exists($this->dbPath),
            'size' => File::exists($this->dbPath) ? File::size($this->dbPath) : 0,
            'permissions' => File::exists($this->dbPath) ? substr(sprintf('%o', fileperms($this->dbPath)), -4) : null,
        ];

        // إضافة معلومات الحماية (بدون كشف كلمة المرور)
        if (File::exists($this->dbPath)) {
            $info['protected'] = true;
            $info['encrypted_data'] = true;
            $info['password_set'] = !empty($this->dbPassword);
        }

        return $info;
    }

    /**
     * الحصول على كلمة مرور قاعدة البيانات (للاستخدام الداخلي فقط)
     */
    public function getPassword(): string
    {
        return $this->dbPassword;
    }

    /**
     * التحقق من كلمة المرور
     */
    public function verifyPassword(string $password): bool
    {
        return hash_equals($this->dbPassword, $password);
    }
}

