<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\Setting;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Only migrate if multiple warehouses mode is enabled
        $warehouseMode = Setting::get('warehouse_mode', 'single');
        
        if ($warehouseMode === 'multiple') {
            // Get or create default warehouse
            $defaultWarehouse = Warehouse::where('is_default', true)->first();
            
            if (!$defaultWarehouse) {
                // Create default warehouse if none exists
                $defaultWarehouse = Warehouse::create([
                    'name' => 'Main Warehouse',
                    'name_ar' => 'المستودع الرئيسي',
                    'code' => 'MAIN',
                    'is_default' => true,
                    'is_active' => true,
                ]);
            }
            
            // Migrate stock from products.stock_quantity to product_warehouse
            $products = Product::all();
            
            foreach ($products as $product) {
                // Check if pivot already exists
                $existingPivot = DB::table('product_warehouse')
                    ->where('product_id', $product->id)
                    ->where('warehouse_id', $defaultWarehouse->id)
                    ->first();
                
                if (!$existingPivot) {
                    // Create pivot record with current stock_quantity
                    DB::table('product_warehouse')->insert([
                        'product_id' => $product->id,
                        'warehouse_id' => $defaultWarehouse->id,
                        'quantity' => $product->stock_quantity ?? 0,
                        'reorder_level' => $product->reorder_level ?? 0,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                } else {
                    // Update existing pivot if quantity is 0
                    if ($existingPivot->quantity == 0 && $product->stock_quantity > 0) {
                        DB::table('product_warehouse')
                            ->where('product_id', $product->id)
                            ->where('warehouse_id', $defaultWarehouse->id)
                            ->update([
                                'quantity' => $product->stock_quantity,
                                'reorder_level' => $product->reorder_level ?? 0,
                                'updated_at' => now(),
                            ]);
                    }
                }
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // This migration is not reversible as we can't determine
        // which warehouse's stock should be restored to products.stock_quantity
        // The user should handle this manually if needed
    }
};
