<!-- Documents Section -->
<div class="card mt-3">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0"><i class="bi bi-folder"></i> {{ trans('messages.documents') }}</h5>
        <button type="button" class="btn btn-sm btn-primary" data-bs-toggle="modal" data-bs-target="#uploadDocumentModal">
            <i class="bi bi-upload"></i> {{ trans('messages.upload_document') }}
        </button>
    </div>
    <div class="card-body">
        <div id="documents-list">
            <div class="text-center py-3">
                <div class="spinner-border text-primary" role="status">
                    <span class="visually-hidden">{{ trans('messages.loading') }}...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Upload Document Modal -->
<div class="modal fade" id="uploadDocumentModal" tabindex="-1" aria-labelledby="uploadDocumentModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <form id="uploadDocumentForm" enctype="multipart/form-data">
                @csrf
                <input type="hidden" name="type" value="{{ $type }}">
                <input type="hidden" name="id" value="{{ $id }}">
                <div class="modal-header">
                    <h5 class="modal-title" id="uploadDocumentModalLabel">{{ trans('messages.upload_document') }}</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label for="document_file" class="form-label">{{ trans('messages.file') }} *</label>
                        <input type="file" class="form-control" id="document_file" name="file" required accept=".pdf,.doc,.docx,.xls,.xlsx,.jpg,.jpeg,.png,.txt">
                        <small class="text-muted">{{ trans('messages.max_file_size') }}: 10MB</small>
                    </div>
                    <div class="mb-3">
                        <label for="document_description" class="form-label">{{ trans('messages.description') }}</label>
                        <textarea class="form-control" id="document_description" name="description" rows="3" placeholder="{{ trans('messages.optional_description') }}"></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">{{ trans('messages.cancel') }}</button>
                    <button type="submit" class="btn btn-primary">{{ trans('messages.upload') }}</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const type = '{{ $type }}';
    const id = {{ $id }};
    
    // Format file size
    function formatFileSize(bytes) {
        if (!bytes) return '0 B';
        const units = ['B', 'KB', 'MB', 'GB'];
        let size = bytes;
        let unit = 0;
        while (size >= 1024 && unit < units.length - 1) {
            size /= 1024;
            unit++;
        }
        return Math.round(size * 100) / 100 + ' ' + units[unit];
    }
    
    // Load documents
    function loadDocuments() {
        fetch(`{{ route('documents.index') }}?type=${type}&id=${id}`)
            .then(response => response.json())
            .then(data => {
                const container = document.getElementById('documents-list');
                if (data.length === 0) {
                    container.innerHTML = `
                        <div class="text-center py-4">
                            <i class="bi bi-folder-x" style="font-size: 3rem; color: #ccc;"></i>
                            <p class="text-muted mt-2">{{ trans('messages.no_documents_found') }}</p>
                        </div>
                    `;
                } else {
                    container.innerHTML = `
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>{{ trans('messages.name') }}</th>
                                        <th>{{ trans('messages.description') }}</th>
                                        <th>{{ trans('messages.size') }}</th>
                                        <th>{{ trans('messages.uploaded_by') }}</th>
                                        <th>{{ trans('messages.date') }}</th>
                                        <th>{{ trans('messages.actions') }}</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    ${data.map(doc => {
                                        const icon = doc.mime_type ? (
                                            doc.mime_type.startsWith('image/') ? 'bi-file-image' :
                                            doc.mime_type === 'application/pdf' ? 'bi-file-pdf' :
                                            doc.mime_type.includes('word') ? 'bi-file-word' :
                                            doc.mime_type.includes('excel') || doc.mime_type.includes('spreadsheet') ? 'bi-file-excel' :
                                            'bi-file-earmark'
                                        ) : 'bi-file-earmark';
                                        
                                        const formattedSize = doc.file_size ? formatFileSize(doc.file_size) : '-';
                                        
                                        return `
                                        <tr>
                                            <td>
                                                <i class="bi ${icon}"></i>
                                                ${doc.name}
                                            </td>
                                            <td>${doc.description || '-'}</td>
                                            <td>${formattedSize}</td>
                                            <td>${doc.uploader ? doc.uploader.name : '-'}</td>
                                            <td>${new Date(doc.created_at).toLocaleDateString()}</td>
                                            <td>
                                                <a href="/documents/${doc.id}/download" class="btn btn-sm btn-primary" title="{{ trans('messages.download') }}">
                                                    <i class="bi bi-download"></i>
                                                </a>
                                                ${doc.mime_type && (doc.mime_type.startsWith('image/') || doc.mime_type === 'application/pdf') ? `
                                                    <a href="/documents/${doc.id}/view" target="_blank" class="btn btn-sm btn-info" title="{{ trans('messages.view') }}">
                                                        <i class="bi bi-eye"></i>
                                                    </a>
                                                ` : ''}
                                                <form action="/documents/${doc.id}" method="POST" class="d-inline" onsubmit="return confirm('{{ trans('messages.confirm_delete_document') }}');">
                                                    @csrf
                                                    @method('DELETE')
                                                    <button type="submit" class="btn btn-sm btn-danger" title="{{ trans('messages.delete') }}">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </form>
                                            </td>
                                        </tr>
                                        `;
                                    }).join('')}
                                </tbody>
                            </table>
                        </div>
                    `;
                }
            })
            .catch(error => {
                console.error('Error loading documents:', error);
                document.getElementById('documents-list').innerHTML = `
                    <div class="alert alert-danger">{{ trans('messages.error_loading_documents') }}</div>
                `;
            });
    }
    
    // Upload document
    document.getElementById('uploadDocumentForm').addEventListener('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const submitBtn = this.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> {{ trans('messages.uploading') }}...';
        
        fetch('{{ route('documents.store') }}', {
            method: 'POST',
            body: formData,
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(async response => {
            const contentType = response.headers.get('content-type');
            if (!contentType || !contentType.includes('application/json')) {
                // If response is not JSON, it might be an HTML error page
                const text = await response.text();
                console.error('Non-JSON response:', text);
                throw new Error('{{ trans('messages.error_uploading_document') }}: Server returned non-JSON response');
            }
            
            const data = await response.json();
            
            if (!response.ok) {
                throw new Error(data.message || data.errors ? JSON.stringify(data.errors) : 'Upload failed');
            }
            
            return data;
        })
        .then(data => {
            if (data.success) {
                // Close modal
                const modal = bootstrap.Modal.getInstance(document.getElementById('uploadDocumentModal'));
                modal.hide();
                
                // Reset form
                this.reset();
                
                // Reload documents
                loadDocuments();
                
                // Show success message
                if (typeof showAlert === 'function') {
                    showAlert('success', data.message || '{{ trans('messages.document_uploaded_successfully') }}');
                } else {
                    alert(data.message || '{{ trans('messages.document_uploaded_successfully') }}');
                }
            } else {
                throw new Error(data.message || 'Upload failed');
            }
        })
        .catch(error => {
            console.error('Error uploading document:', error);
            let errorMessage = '{{ trans('messages.error_uploading_document') }}';
            if (error.message) {
                errorMessage += ': ' + error.message;
            }
            if (typeof showAlert === 'function') {
                showAlert('danger', errorMessage);
            } else {
                alert(errorMessage);
            }
        })
        .finally(() => {
            submitBtn.disabled = false;
            submitBtn.innerHTML = originalText;
        });
    });
    
    // Load documents on page load
    loadDocuments();
    
    // Reload documents when modal is closed (in case of successful upload)
    document.getElementById('uploadDocumentModal').addEventListener('hidden.bs.modal', function() {
        loadDocuments();
    });
});
</script>

