@extends('layouts.master')

@section('title', trans('messages.create_quotation'))

@section('content')
<div class="page-header">
    <h2>{{ trans('messages.create_quotation') }}</h2>
    <a href="{{ route('quotations.index') }}" class="btn btn-secondary">{{ trans('messages.back') }}</a>
</div>

<div class="card">
    <div class="card-body">
        <form method="POST" action="{{ route('quotations.store') }}" id="quotation-form">
            @csrf
            <div class="row mb-3">
                <div class="col-md-6">
                    <label for="customer_id" class="form-label">{{ trans('messages.customer') }}</label>
                    <select class="form-select" id="customer_id" name="customer_id">
                        <option value="">{{ trans('messages.walk_in_customer') }}</option>
                        @foreach($customers as $customer)
                            <option value="{{ $customer->id }}">{{ $customer->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-6">
                    <label for="customer_name" class="form-label">{{ trans('messages.customer_name') }} ({{ trans('messages.walk_in_customer') }})</label>
                    <input type="text" class="form-control" id="customer_name" name="customer_name" placeholder="{{ trans('messages.walk_in_customer') }}">
                </div>
            </div>

            <div class="mb-3">
                <label class="form-label">{{ trans('messages.items') }}</label>
                <div class="table-responsive">
                    <table class="table table-bordered" id="items-table">
                        <thead>
                            <tr>
                                <th>{{ trans('messages.product') }}</th>
                                <th>{{ trans('messages.quantity') }}</th>
                                <th>{{ trans('messages.unit_price') }}</th>
                                <th>{{ trans('messages.tax_rate') }} (%)</th>
                                <th>{{ trans('messages.discount') }}</th>
                                <th>{{ trans('messages.total') }}</th>
                                <th>{{ trans('messages.actions') }}</th>
                            </tr>
                        </thead>
                        <tbody id="items-tbody">
                            <!-- Items will be added dynamically -->
                        </tbody>
                        <tfoot>
                            <tr>
                                <td colspan="5" class="text-end"><strong>{{ trans('messages.subtotal') }}:</strong></td>
                                <td><strong id="subtotal">0.00</strong></td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>{{ trans('messages.discount') }}:</strong></td>
                                <td><input type="number" class="form-control" id="discount" name="discount" value="0" step="0.01" min="0"></td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>{{ trans('messages.tax') }}:</strong></td>
                                <td><strong id="tax">0.00</strong></td>
                                <td></td>
                            </tr>
                            <tr>
                                <td colspan="5" class="text-end"><strong>{{ trans('messages.total') }}:</strong></td>
                                <td><strong id="total">0.00</strong></td>
                                <td></td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                <button type="button" class="btn btn-sm btn-primary" id="add-item-btn">
                    <i class="bi bi-plus-circle"></i> {{ trans('messages.add_item') }}
                </button>
            </div>

            <div class="mb-3">
                <label for="notes" class="form-label">{{ trans('messages.notes') }}</label>
                <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
            </div>

            <div class="d-flex gap-2">
                <button type="submit" class="btn btn-primary">{{ trans('messages.save') }}</button>
                <a href="{{ route('quotations.index') }}" class="btn btn-secondary">{{ trans('messages.cancel') }}</a>
            </div>
        </form>
    </div>
</div>

@push('scripts')
<script>
@php
    $productsArray = [];
    foreach ($products as $product) {
        $productsArray[] = [
            'id' => $product->id,
            'name' => $product->name,
            'selling_price' => $product->selling_price,
            'tax_rate' => $product->tax_rate ?? 0,
            'stock_quantity' => $product->stock_quantity,
        ];
    }
@endphp
const products = @json($productsArray);

let itemIndex = 0;

document.getElementById('add-item-btn').addEventListener('click', function() {
    addItemRow();
});

function addItemRow() {
    const tbody = document.getElementById('items-tbody');
    const row = document.createElement('tr');
    row.dataset.index = itemIndex;
    
    const productSelect = document.createElement('select');
    productSelect.className = 'form-select product-select';
    productSelect.name = `items[${itemIndex}][product_id]`;
    productSelect.required = true;
    productSelect.innerHTML = '<option value="">{{ trans("messages.select_product") }}</option>';
    products.forEach(product => {
        const option = document.createElement('option');
        option.value = product.id;
        option.textContent = product.name;
        option.dataset.price = product.selling_price;
        option.dataset.tax = product.tax_rate;
        productSelect.appendChild(option);
    });

    row.innerHTML = `
        <td>${productSelect.outerHTML}</td>
        <td><input type="number" class="form-control quantity" name="items[${itemIndex}][quantity]" value="1" min="1" required></td>
        <td><input type="number" class="form-control unit-price" name="items[${itemIndex}][unit_price]" step="0.01" min="0" required></td>
        <td><input type="number" class="form-control tax-rate" name="items[${itemIndex}][tax_rate]" step="0.01" min="0" max="100" value="0"></td>
        <td><input type="number" class="form-control discount" name="items[${itemIndex}][discount]" step="0.01" min="0" value="0"></td>
        <td><span class="item-total">0.00</span></td>
        <td><button type="button" class="btn btn-sm btn-danger remove-item"><i class="bi bi-trash"></i></button></td>
    `;
    
    tbody.appendChild(row);
    
    // Add event listeners
    const select = row.querySelector('.product-select');
    select.addEventListener('change', function() {
        const option = this.options[this.selectedIndex];
        if (option.value) {
            row.querySelector('.unit-price').value = option.dataset.price;
            row.querySelector('.tax-rate').value = option.dataset.tax;
            calculateItemTotal(row);
        }
    });
    
    row.querySelectorAll('.quantity, .unit-price, .tax-rate, .discount').forEach(input => {
        input.addEventListener('input', () => calculateItemTotal(row));
    });
    
    row.querySelector('.remove-item').addEventListener('click', function() {
        row.remove();
        calculateTotals();
    });
    
    itemIndex++;
}

function calculateItemTotal(row) {
    const quantity = parseFloat(row.querySelector('.quantity').value) || 0;
    const unitPrice = parseFloat(row.querySelector('.unit-price').value) || 0;
    const taxRate = parseFloat(row.querySelector('.tax-rate').value) || 0;
    const discount = parseFloat(row.querySelector('.discount').value) || 0;
    
    const subtotal = quantity * unitPrice;
    const tax = subtotal * (taxRate / 100);
    const total = subtotal + tax - discount;
    
    row.querySelector('.item-total').textContent = total.toFixed(2);
    calculateTotals();
}

function calculateTotals() {
    let subtotal = 0;
    let tax = 0;
    
    document.querySelectorAll('#items-tbody tr').forEach(row => {
        const quantity = parseFloat(row.querySelector('.quantity').value) || 0;
        const unitPrice = parseFloat(row.querySelector('.unit-price').value) || 0;
        const taxRate = parseFloat(row.querySelector('.tax-rate').value) || 0;
        const discount = parseFloat(row.querySelector('.discount').value) || 0;
        
        const itemSubtotal = quantity * unitPrice;
        const itemTax = itemSubtotal * (taxRate / 100);
        
        subtotal += itemSubtotal - discount;
        tax += itemTax;
    });
    
    const discount = parseFloat(document.getElementById('discount').value) || 0;
    const total = subtotal + tax - discount;
    
    document.getElementById('subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('tax').textContent = tax.toFixed(2);
    document.getElementById('total').textContent = total.toFixed(2);
}

document.getElementById('discount').addEventListener('input', calculateTotals);

// Add first item row on page load
addItemRow();
</script>
@endpush
@endsection

