<?php

namespace Tests\Feature;

use App\Models\Invoice;
use App\Models\Product;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class InvoiceTest extends TestCase
{
    use RefreshDatabase;

    public function test_user_can_create_invoice(): void
    {
        $user = User::factory()->create();
        $user->assignRole('Cashier');

        $product = Product::factory()->create(['stock_quantity' => 100]);

        $response = $this->actingAs($user)->post('/invoices', [
            'customer_name' => 'Test Customer',
            'items' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 5,
                    'unit_price' => $product->selling_price,
                    'tax_rate' => 10,
                    'discount' => 0,
                ],
            ],
            'discount' => 0,
        ]);

        $response->assertRedirect();
        $this->assertDatabaseHas('invoices', ['customer_name' => 'Test Customer']);
    }

    public function test_finalizing_invoice_reduces_stock(): void
    {
        $user = User::factory()->create();
        $user->assignRole('Cashier');

        $product = Product::factory()->create(['stock_quantity' => 100]);
        $invoice = Invoice::factory()->create(['status' => 'draft', 'user_id' => $user->id]);
        $invoice->items()->create([
            'product_id' => $product->id,
            'quantity' => 10,
            'unit_price' => 10,
            'tax_rate' => 0,
            'discount' => 0,
            'subtotal' => 100,
            'total' => 100,
        ]);

        $response = $this->actingAs($user)->post("/invoices/{$invoice->id}/finalize");

        $response->assertRedirect();
        $this->assertEquals(90, $product->fresh()->stock_quantity);
    }
}

